# Essential v1.0 Features and JJ Commands

This document maps the v1.0 essential features to the specific `jj` commands needed for implementation.

## Repository Detection

**Requirement**: Automatically detect existing jj repositories and register VCS roots

### Commands Needed:

- `jj root` - Check if directory is in a jj repository and get repository root path

**Implementation Notes**:

- Run `jj root` to both detect if directory contains a jj repo (exit code) and get canonical repository path for VCS
  root registration
- Multi-workspace support deferred to future versions

## File Status Tracking

**Requirement**: Show modified/added/deleted files in Changes view and file tree

### Commands Needed:

- `jj status --no-pager --color=never -T TEMPLATE` - Get working copy file changes with structured output

**Template for JSON-like output**:

```bash
jj status -T 'format_file_status(files)'
# Or custom template for kotlinx.serialization compatibility
```

**Implementation Notes**:

- Use jj templating language to generate JSON-compatible output for kotlinx.serialization
- Avoid parsing default jj output format - use templates for structured data
- Map jj file states to IntelliJ's FileStatus enum (MODIFIED, ADDED, DELETED, etc.)
- Update file tree icons and Changes view based on status

## Basic Commit Operations

**Requirement**: Create new changes via standard commit dialog

### Commands Needed:

- `jj commit -m "commit message"` - Create new change with message

**Implementation Notes**:

- Map IntelliJ's commit dialog directly to `jj commit` with message
- To opt into IntelliJ's non-modal commit tool, mark AbstractVcs implementation as DVCS
- Partial file commits and interactive staging deferred to future versions

## Diff Visualization

**Requirement**: View file differences using IntelliJ's built-in diff viewer

### Commands Needed:

- `jj diff` - Show changes in working copy
- `jj diff --revision REV` - Show changes in specific revision
- `jj diff --from REV --to REV` - Show diff between two revisions
- `jj diff --tool :dump` - Get diff in parseable format
- `jj show REV` - Show specific change content

**File-specific diffs**:

- `jj diff path/to/file` - Diff for specific file
- `jj diff --revision REV path/to/file` - File diff for specific revision

**Implementation Notes**:

- Parse diff output and feed to IntelliJ's diff viewer
- Support both working copy and historical diffs

## Change History

**Requirement**: Browse change history with basic log view

### Commands Needed:

- `jj log` - Show change history
- `jj log --template TEMPLATE` - Formatted log output
- `jj log --limit N` - Limit number of changes shown
- `jj log --revisions REVSET` - Filter changes by revset

**Useful templates**:

```bash
jj log -T 'commit_id.short() ++ " " ++ description.first_line() ++ "\n"'
jj log -T 'format_detailed_log(self)'
```

**Metadata queries**:

- `jj show --template TEMPLATE REV` - Get detailed change info
- `jj log --template 'format_timestamp(committer.timestamp())' -r REV` - Get timestamps

**Implementation Notes**:

- Map jj changes to IntelliJ's VcsCommit model
- Use templates for structured data extraction
- Support filtering and searching in log view

## Working Copy Status

**Requirement**: Show current working copy state and conflicts

### Commands Needed:

- `jj status` - Overall working copy status
- `jj resolve --list` - List unresolved conflicts
- `jj log -r @ -T 'description'` - Current change description
- `jj log -r @ -T 'change_id'` - Current change ID

**Conflict information**:

- `jj resolve --list --quiet` - Machine-readable conflict list
- `jj diff --conflicts` - Show conflict markers in files

**Branch/bookmark status**:

- `jj bookmark list` - List local bookmarks
- `jj log -r @ -T 'bookmarks'` - Bookmarks on current change

**Implementation Notes**:

- Display working copy status in IntelliJ's status bar
- Show conflict indicators in file tree and editor
- Integrate with IntelliJ's VCS status display

## Additional Utility Commands

### Repository Information:

- `jj workspace root` - Get workspace root (alternative to `jj root`)
- `jj config list` - Get repository configuration
- `jj operation log` - Show operation history (for debugging)

### Error Handling:

- All commands should use `--quiet` flag where available to reduce noise
- Use `--color=never` to ensure parseable output
- Check exit codes to detect command failures

### Performance Considerations:

- Use `--limit` on log commands for large repositories
- Consider caching frequently-used command outputs
- Run status checks asynchronously to avoid UI blocking

## Testing Strategy

### Test Categories

**Light Tests (Recommended Default)**:

- Use `LightPlatformTestCase` or `BasePlatformTestCase`
- Reuse projects between test runs for better performance
- Test most plugin functionality including command parsing and VCS integration

**Heavy Tests (Complex Scenarios Only)**:

- Use `HeavyPlatformTestCase`
- Create new project per test
- Required for multi-workspace or complex repository scenarios

### Unit Tests

**Command Execution & Parsing**:

- Mock `jj` command execution using process builders
- Test templated output parsing with kotlinx.serialization
- Verify mapping between jj file states and IntelliJ FileStatus enum
- Test error handling for invalid repositories and command failures

**VCS Framework Integration**:

- Test `ChangeProvider` correctly identifies modified files from `jj status`
- Verify repository detection logic with `jj root`
- Test commit operations execute correct `jj commit` commands
- Mock file system operations for repository detection tests

### Integration Tests

**End-to-End Workflows**:

- Create test jj repositories using `jj init`
- Test complete edit → commit → history workflows
- Verify Changes view and file tree status indicators
- Test diff visualization with real jj repository changes

**IDE Integration**:

- Test VCS root registration and detection
- Verify non-modal commit dialog integration (DVCS marking)
- Test performance with larger test repositories
- Verify cross-IDE compatibility (IntelliJ, PyCharm, WebStorm)

### Test Implementation Notes

**Key Testing Utilities**:

- `CodeInsightTestFixture` - For testing VCS UI integration
- Mock jj command execution to avoid external dependencies
- Use temporary directories for repository creation tests
- Test with various jj repository states (clean, modified, conflicted)
