/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.execution.util.ExecUtil
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.vcs.VcsException
import java.io.Closeable
import java.io.File
import java.nio.file.Files
import java.nio.file.Path
import java.security.SecureRandom

/**
 * Manages the lifecycle of a fake diff editor for jj tool integration, providing communication
 * between the plugin and the script for interactive commits.
 *
 * This class implements Closeable to ensure proper resource cleanup.
 * If closed without explicitly setting an exit code, it will signal the script to exit with code 1.
 */
class FakeDiffEditor : Closeable {

    companion object {
        private val LOG = logger<FakeDiffEditor>()
    }

    private val inputFile: Path = Files.createTempFile("selvejj_input_", ".txt")
    private val outputFile: Path = Files.createTempFile("selvejj_output_", ".txt")
    private val sentinelString: String = generateRandomSentinel()
    private val scriptFile: File
    private var process: Process? = null
    private var exitCode: Int = 1
    private var closed: Boolean = false

    init {
        val isWindows = System.getProperty("os.name").lowercase().contains("windows")
        val scriptSuffix = if (isWindows) ".bat" else ".sh"

        scriptFile = ExecUtil.createTempExecutableScript(
            prefix = "selvejj_commit_tool_",
            suffix = scriptSuffix,
            content = createCommitToolScript(inputFile, outputFile, sentinelString)
        )

        LOG.debug("Created FakeDiffEditor with files: input=${inputFile}, output=${outputFile}, script=${scriptFile.absolutePath}, sentinel=${sentinelString}")
    }

    /**
     * Gets the path to the script file that should be passed to jj's --tool parameter
     */
    val scriptPath: String
        get() {
            checkNotClosed()
            return scriptFile.absolutePath
        }

    /**
     * Starts the jj process with the given command and waits for the script to be ready.
     * Returns a pair of (leftPath, rightPath) that the script received from jj.
     */
    fun startAndWaitForReady(command: GeneralCommandLine): Pair<String, String> {
        checkNotClosed()

        if (process != null) {
            throw IllegalStateException("Process already started")
        }

        // Start jj commit in background
        process = command.createProcess()
        LOG.debug("Started jj process")

        // Wait for script to write left and right paths
        val (leftPath, rightPath) = waitForScriptCommunication(inputFile, sentinelString)
        LOG.debug("Received paths from script: left=${leftPath}, right=${rightPath}")

        return Pair(leftPath, rightPath)
    }

    /**
     * Sets the exit code that the script should return.
     * The actual signaling and process completion happens in close().
     */
    fun setExitCode(exitCode: Int) {
        checkNotClosed()
        this.exitCode = exitCode
        LOG.debug("Set exit code: ${exitCode}")
    }

    override fun close() {
        if (closed) {
            return
        }

        closed = true

        try {
            // Signal the script with the exit code
            process?.let { proc ->
                try {
                    signalScriptExit(outputFile, sentinelString, exitCode)
                    LOG.debug("Signaled script to exit with code ${exitCode}")

                    val actualExitCode = proc.waitFor()
                    LOG.debug("jj process completed with exit code: ${actualExitCode}")
                } catch (e: Exception) {
                    LOG.warn("Error while signaling or waiting for process completion", e)
                }

                // Clean up process if still alive
                if (proc.isAlive) {
                    proc.destroyForcibly()
                    LOG.debug("Forcibly destroyed jj process")
                }
            }
        } finally {
            // Clean up files
            try {
                scriptFile.delete()
                Files.deleteIfExists(inputFile)
                Files.deleteIfExists(outputFile)
                LOG.debug("Cleaned up temporary files")
            } catch (e: Exception) {
                LOG.warn("Failed to clean up temporary files", e)
            }
        }
    }

    private fun checkNotClosed() {
        if (closed) {
            throw IllegalStateException("FakeDiffEditor has been closed")
        }
    }

    private fun generateRandomSentinel(): String {
        val random = SecureRandom()
        val bytes = ByteArray(16)
        random.nextBytes(bytes)
        return bytes.joinToString("") { "%02x".format(it) }
    }

    private fun createCommitToolScript(inputFile: Path, outputFile: Path, sentinelString: String): String {
        val isWindows = System.getProperty("os.name").lowercase().contains("windows")

        return if (isWindows) {
            """
                @echo off
                setlocal enabledelayedexpansion
                
                set "left=%1"
                set "right=%2"
                
                rem Write left and right paths to input file, followed by sentinel
                echo !left! > "${inputFile.toAbsolutePath()}"
                echo !right! >> "${inputFile.toAbsolutePath()}"
                echo ${sentinelString} >> "${inputFile.toAbsolutePath()}"
                
                rem Wait for exit signal from plugin
                :wait_loop
                if exist "${outputFile.toAbsolutePath()}" (
                    findstr /c:"${sentinelString}" "${outputFile.toAbsolutePath()}" >nul 2>&1
                    if !errorlevel! equ 0 (
                        for /f %%i in ('type "${outputFile.toAbsolutePath()}"') do (
                            exit %%i
                            goto :eof
                        )
                    )
                )
                timeout /t 0 >nul 2>&1
                goto wait_loop
            """.trimIndent()
        } else {
            """
                #!/bin/bash
                set -euo pipefail
                
                left="${'$'}1"
                right="${'$'}2"
                
                # Write left and right paths to input file, followed by sentinel
                echo "${'$'}left" > "${inputFile.toAbsolutePath()}"
                echo "${'$'}right" >> "${inputFile.toAbsolutePath()}"
                echo "${sentinelString}" >> "${inputFile.toAbsolutePath()}"
                
                # Wait for exit signal from plugin
                while true; do
                    if [ -f "${outputFile.toAbsolutePath()}" ] && grep -q "${sentinelString}" "${outputFile.toAbsolutePath()}"; then
                        # Read exit code from output file
                        exit_code=${'$'}(head -1 "${outputFile.toAbsolutePath()}")
                        exit "${'$'}exit_code"
                    fi
                    sleep 0.1
                done
            """.trimIndent()
        }
    }

    private fun waitForScriptCommunication(inputFile: Path, sentinelString: String): Pair<String, String> {
        val maxWaitTime = 30000 // 30 seconds timeout
        val startTime = System.currentTimeMillis()

        while (System.currentTimeMillis() - startTime < maxWaitTime) {
            if (Files.exists(inputFile)) {
                try {
                    val lines = Files.readAllLines(inputFile)
                    if (lines.size >= 3 && lines[2] == sentinelString) {
                        return Pair(lines[0], lines[1])
                    }
                } catch (e: Exception) {
                    // File might be partially written, continue waiting
                }
            }
            Thread.sleep(100)
        }
        throw VcsException("Timeout waiting for script communication")
    }

    private fun signalScriptExit(outputFile: Path, sentinelString: String, exitCode: Int) {
        Files.write(outputFile, listOf(exitCode.toString(), sentinelString))
    }
}