/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.execution.util.ExecUtil
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.progress.ProgressIndicator
import com.intellij.openapi.vcs.VcsException
import com.intellij.openapi.vcs.changes.*
import com.intellij.openapi.vcs.FilePath
import com.intellij.vcsUtil.VcsUtil
import java.io.File

class SelvejjChangeProvider : ChangeProvider {

    companion object {
        private val LOG = logger<SelvejjChangeProvider>()
    }

    override fun getChanges(
        dirtyScope: VcsDirtyScope,
        builder: ChangelistBuilder,
        progress: ProgressIndicator,
        addGate: ChangeListManagerGate
    ) {
        try {
            val repositoryRoot = getRepositoryRoot(dirtyScope) ?: return
            val filesets = buildFilesets(dirtyScope, repositoryRoot)

            if (filesets.isEmpty()) return

            // Get canonical commit IDs
            val parentCommitId = getCommitId(repositoryRoot, "@-")

            if (parentCommitId == null) {
                LOG.warn("Failed to get canonical commit IDs")
                return
            }

            val changes = getJjChanges(repositoryRoot, filesets)

            for (changeInfo in changes) {
                val change = createChange(changeInfo, repositoryRoot, parentCommitId)
                if (change != null) {
                    builder.processChange(change, SelvejjVcs.getKey())
                }
            }
        } catch (e: Exception) {
            LOG.warn("Error getting changes from jj", e)
            throw VcsException("Failed to get changes from Jujutsu: ${e.message}", e)
        }
    }

    override fun isModifiedDocumentTrackingRequired(): Boolean {
        return true
    }

    private fun getRepositoryRoot(dirtyScope: VcsDirtyScope): String? {
        val project = dirtyScope.project
        val projectPath = project.basePath ?: return null

        val command = GeneralCommandLine("jj")
            .withParameters("workspace", "root")
            .withWorkDirectory(projectPath)

        val result = ExecUtil.execAndGetOutput(command, 5000)
        return if (result.checkSuccess(LOG)) result.stdout.trim() else null
    }

    private fun buildFilesets(dirtyScope: VcsDirtyScope, repositoryRoot: String): List<String> {
        val filesets = mutableListOf<String>()

        // Add dirty files
        dirtyScope.dirtyFiles.forEach { file ->
            val relativePath = getRelativePath(file.path, repositoryRoot)
            if (relativePath != null) {
                filesets.add("root:\"$relativePath\"")
            }
        }

        // Add recursively dirty directories
        dirtyScope.recursivelyDirtyDirectories.forEach { dir ->
            val relativePath = getRelativePath(dir.path, repositoryRoot)
            if (relativePath != null) {
                filesets.add("root:\"$relativePath\"")
            }
        }

        return filesets
    }

    private fun getRelativePath(absolutePath: String, repositoryRoot: String): String? {
        val file = File(absolutePath)
        val root = File(repositoryRoot)

        return if (file.startsWith(root)) {
            root.toPath().relativize(file.toPath()).toString()
        } else {
            null
        }
    }

    private fun getJjChanges(repositoryRoot: String, filesets: List<String>): List<JjChangeInfo> {
        val template =
            "diff.files().map(|f| f.status() ++ \":\" ++ f.source().path() ++ \":\" ++ f.target().path() ++ \":\" ++ f.source().file_type() ++ \":\" ++ f.target().file_type()).join(\"\\n\")"

        val command = GeneralCommandLine("jj")
            .withParameters("log", "-R", repositoryRoot, "--no-graph", "-r", "@", "-T", template)
            .withParameters(filesets)

        val result = ExecUtil.execAndGetOutput(command, 10000)
        if (!result.checkSuccess(LOG)) {
            throw VcsException("jj log command failed: ${result.stderr}")
        }

        return parseJjOutput(result.stdout.trim())
    }

    private fun parseJjOutput(output: String): List<JjChangeInfo> {
        if (output.isEmpty()) return emptyList()

        return output.split('\n').mapNotNull { line ->
            val parts = line.split(':', limit = 5)
            if (parts.size == 5) {
                JjChangeInfo(
                    status = parts[0],
                    sourcePath = parts[1].takeIf { it.isNotEmpty() },
                    targetPath = parts[2].takeIf { it.isNotEmpty() },
                    sourceFileType = parts[3].takeIf { it.isNotEmpty() },
                    targetFileType = parts[4].takeIf { it.isNotEmpty() }
                )
            } else {
                LOG.warn("Unexpected jj output format: $line")
                null
            }
        }
    }

    private fun getCommitId(repositoryRoot: String, revset: String): String? {
        val command = GeneralCommandLine("jj")
            .withParameters("log", "-R", repositoryRoot, "-r", revset, "-T", "commit_id", "--no-graph")

        val result = ExecUtil.execAndGetOutput(command, 5000)
        return if (result.checkSuccess(LOG)) result.stdout.trim() else null
    }

    private fun createChange(
        changeInfo: JjChangeInfo,
        repositoryRoot: String,
        parentCommitId: String,
    ): Change? {
        val beforeRevision = when (changeInfo.status) {
            "added" -> null // Added file - no before revision
            else -> createContentRevision(
                changeInfo.sourcePath,
                changeInfo.sourceFileType,
                repositoryRoot,
                parentCommitId
            )
        }

        val afterRevision = when (changeInfo.status) {
            "removed" -> null // Deleted file - no after revision  
            else -> createContentRevision(changeInfo.targetPath, changeInfo.targetFileType, repositoryRoot, null)
        }

        return if (beforeRevision != null || afterRevision != null) {
            Change(beforeRevision, afterRevision)
        } else {
            null
        }
    }

    private fun createContentRevision(
        path: String?,
        fileType: String?,
        repositoryRoot: String,
        commitId: String?
    ): ContentRevision? {
        if (path == null) return null

        val isDirectory = fileType == "tree"
        val filePath = VcsUtil.getFilePath(File(repositoryRoot, path), isDirectory)

        return if (commitId != null) {
            // For before revision, use historical content with canonical commit ID
            SelvejjHistoricalContentRevision(
                filePath = filePath,
                revisionNumber = SelvejjRevisionNumber(commitId),
                repositoryRoot = repositoryRoot,
                relativePath = path
            )
        } else {
            // For after revision, get current working copy content
            SelvejjContentRevision(filePath)
        }
    }

    private data class JjChangeInfo(
        val status: String,
        val sourcePath: String?,
        val targetPath: String?,
        val sourceFileType: String?,
        val targetFileType: String?
    )
}