/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.dvcs.repo.Repository
import com.intellij.dvcs.repo.RepositoryImpl
import com.intellij.openapi.Disposable
import com.intellij.openapi.project.Project
import com.intellij.openapi.util.Disposer
import com.intellij.openapi.vcs.ProjectLevelVcsManager
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.util.messages.Topic
import java.io.File

/**
 * Implementation of [Repository] for Jujutsu VCS.
 * Represents a Jujutsu repository and provides access to its state.
 */
class SelvejjRepository(
    project: Project,
    root: VirtualFile,
    parentDisposable: Disposable
) : RepositoryImpl(project, root, parentDisposable) {

    companion object {
        /**
         * Topic for repository change events.
         * Subscribers will be notified when a repository's state changes.
         */
        @JvmField
        val JJ_REPO_CHANGE = Topic.create(
            "SelvejjRepository change",
            SelvejjRepositoryChangeListener::class.java
        )

        /**
         * Path to the Jujutsu working copy checkout file.
         */
        const val JJ_WORKING_COPY_PATH = ".jj/working_copy/checkout"
    }

    /**
     * Current state of the repository.
     */
    @Volatile
    private var myState: Repository.State = Repository.State.NORMAL

    /**
     * Repository updater that monitors file system changes.
     */
    private var updater: SelvejjRepositoryUpdater? = null

    init {
        update()
    }

    /**
     * Sets up the repository updater to monitor file system changes.
     * Should be called after the repository is fully constructed.
     */
    fun setupUpdater() {
        updater = SelvejjRepositoryUpdater(project, this)
        Disposer.register(this, updater!!)
    }

    /**
     * Updates the repository state by reading from disk.
     * This method is called when file changes are detected.
     */
    override fun update() {
        // Read repository state from disk
        // For now, just set state to NORMAL
        myState = Repository.State.NORMAL

        // Publish change event
        project.messageBus.syncPublisher(JJ_REPO_CHANGE).repositoryChanged(this)
    }

    /**
     * Returns the current state of the repository.
     */
    override fun getState(): Repository.State {
        return myState
    }

    /**
     * Returns the VCS type.
     */
    override fun getVcs(): SelvejjVcs {
        return ProjectLevelVcsManager.getInstance(project).findVcsByName(SelvejjVcs.VCS_NAME) as SelvejjVcs
    }

    /**
     * Returns the current branch name.
     * In Jujutsu, this would be the current working copy revision.
     */
    override fun getCurrentBranchName(): String? {
        // TODO: Implement getting current branch name
        return null
    }

    /**
     * Returns the current revision hash.
     * In Jujutsu, this would be the current working copy revision hash.
     */
    override fun getCurrentRevision(): String? {
        // TODO: Implement getting current revision
        return null
    }

    /**
     * Returns a string representation of this repository for logging.
     */
    override fun toLogString(): String {
        return "SelvejjRepository(${root.path})"
    }

    /**
     * Returns the path to the .jj directory.
     */
    fun getJjDir(): File {
        return File(root.path, ".jj")
    }

}