/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj.actions

import com.intellij.openapi.actionSystem.ActionUpdateThread
import com.intellij.openapi.actionSystem.AnActionEvent
import com.intellij.openapi.ide.CopyPasteManager
import com.intellij.openapi.project.DumbAwareAction
import com.intellij.openapi.vcs.actions.ShowAnnotateOperationsPopup
import com.intellij.openapi.vcs.annotate.FileAnnotation
import com.intellij.util.ui.TextTransferable
import com.selvejj.SelvejjAnnotation

/**
 * Action to copy the change ID from an annotation line.
 * This action is shown in the annotation gutter context menu.
 */
class CopyChangeIdFromAnnotateAction(private val annotation: FileAnnotation) : DumbAwareAction("Copy Change ID") {

    override fun getActionUpdateThread(): ActionUpdateThread {
        return ActionUpdateThread.BGT
    }

    override fun actionPerformed(e: AnActionEvent) {
        val lineNumber = ShowAnnotateOperationsPopup.getAnnotationLineNumber(e.dataContext)
        if (lineNumber < 0) return

        if (annotation is SelvejjAnnotation) {
            val changeId = annotation.getChangeId(lineNumber)
            if (changeId != null) {
                CopyPasteManager.getInstance().setContents(TextTransferable(changeId as CharSequence))
            }
        }
    }

    override fun update(e: AnActionEvent) {
        val lineNumber = ShowAnnotateOperationsPopup.getAnnotationLineNumber(e.dataContext)
        val enabled = lineNumber >= 0 && annotation is SelvejjAnnotation
        e.presentation.isEnabledAndVisible = enabled
    }
}
