/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj.ui

import com.intellij.dvcs.ui.DvcsCloneDialogComponent
import com.intellij.openapi.diagnostic.Logger
import com.intellij.openapi.progress.ProgressManager
import com.intellij.openapi.project.Project
import com.intellij.openapi.vcs.CheckoutProvider
import com.intellij.openapi.vcs.VcsNotifier
import com.intellij.openapi.vcs.ui.cloneDialog.VcsCloneDialogComponentStateListener
import com.intellij.openapi.vfs.LocalFileSystem
import com.selvejj.SelvejjRememberedInputs
import java.io.File
import java.nio.file.Paths

/**
 * Clone dialog component for Jujutsu (jj) VCS
 */
class SelvejjCloneDialogComponent(
    project: Project,
    dialogStateListener: VcsCloneDialogComponentStateListener
) : DvcsCloneDialogComponent(
    project,
    ".jj", // jj repository directory name
    SelvejjRememberedInputs.getInstance(),
    dialogStateListener
) {

    private val logger = Logger.getInstance(SelvejjCloneDialogComponent::class.java)

    override fun onComponentSelected(dialogStateListener: VcsCloneDialogComponentStateListener) {
        updateOkActionState(dialogStateListener)
    }

    override fun isOkActionEnabled(): Boolean {
        return super.isOkActionEnabled() && getDirectory().isNotBlank()
    }

    override fun doClone(listener: CheckoutProvider.Listener) {
        val sourceUrl = getUrl()
        val destinationDir = getDirectory()

        if (sourceUrl.isBlank()) {
            logger.warn("Clone failed: empty source URL")
            notifyError("Clone failed: Repository URL is empty")
            return
        }

        if (destinationDir.isBlank()) {
            logger.warn("Clone failed: empty destination directory")
            notifyError("Clone failed: Destination directory is empty")
            return
        }

        val destinationPath = Paths.get(destinationDir)
        val parentDir = destinationPath.parent?.toFile()

        if (parentDir == null || !parentDir.exists()) {
            logger.warn("Clone failed: parent directory does not exist: ${parentDir?.absolutePath}")
            notifyError("Clone failed: Parent directory does not exist")
            return
        }

        val projectName = destinationPath.fileName.toString()

        ProgressManager.getInstance().runProcessWithProgressSynchronously({
            try {
                val processBuilder = ProcessBuilder("jj", "git", "clone", sourceUrl, projectName)
                processBuilder.directory(parentDir)
                processBuilder.redirectErrorStream(true)

                val process = processBuilder.start()
                val exitCode = process.waitFor()

                if (exitCode == 0) {
                    val projectDir = File(parentDir, projectName)
                    val vf = LocalFileSystem.getInstance().refreshAndFindFileByIoFile(projectDir)
                    if (vf != null) {
                        listener.directoryCheckedOut(projectDir, com.selvejj.SelvejjVcs.getKey())
                        listener.checkoutCompleted()

                        // Remember the inputs for next time
                        val rememberedInputs = SelvejjRememberedInputs.getInstance()
                        rememberedInputs.addUrl(sourceUrl)
                        rememberedInputs.cloneParentDir = parentDir.absolutePath
                    } else {
                        notifyError("Clone completed but could not find cloned directory")
                    }
                } else {
                    val errorOutput = process.inputStream.bufferedReader().readText()
                    logger.warn("Clone failed with exit code $exitCode: $errorOutput")
                    notifyError("Clone failed: $errorOutput")
                }
            } catch (e: Exception) {
                logger.error("Clone failed with exception", e)
                notifyError("Clone failed: ${e.message}")
            }
        }, "Cloning Repository...", true, project)
    }

    private fun notifyError(message: String) {
        VcsNotifier.getInstance(project).notifyError(
            "Jujutsu Clone Failed",
            "Jujutsu clone",
            message
        )
    }
}