# IntelliJ Plugin Testing Guide

Quick reference for testing IntelliJ Platform plugins, synthesized from official JetBrains documentation.

---

## Testing Philosophy

IntelliJ Platform uses **"model-level functional tests"** for stability:
- **Complete feature testing** over isolated unit tests
- **Direct model testing** instead of UI testing
- **Real implementations** in headless environment (except UI)
- **Input/output comparison** using source files with markup
- **Minimal mocking** - framework provides real component interactions

**Scale context**: IntelliJ maintains 400,000+ unit tests but only ~1,000 integration tests. Integration tests are reserved for complex UI scenarios, multi-component interactions, and real-world workflow validation that unit tests cannot effectively cover.

---

## Test Types Quick Reference

**When to Use What:**
- **Light Tests** (default) → Fast, reuses projects, single-module scenarios
  - No Java PSI: `BasePlatformTestCase`
  - With Java PSI: `LightJavaCodeInsightFixtureTestCase`
- **Heavy Tests** → Multi-module projects only (`HeavyPlatformTestCase`)
- **Integration Tests** → UI interactions, full IDE testing, classpath/plugin declaration issues

---

## Setting Up Tests

### Unit Tests

```kotlin
dependencies {
    testImplementation("junit:junit:4.13.2") // or JUnit 5
}
```

**Base test class:**
```kotlin
class MyFeatureTest : BasePlatformTestCase() {
    override fun getTestDataPath() = "src/test/testData"

    fun testMyFeature() {
        myFixture.configureByFile("input.kt")
        myFixture.type("text")
        myFixture.checkResultByFile("expected.kt")
    }
}
```

**Critical Setup Rules:**
1. Always call `super.tearDown()` in `finally` block
2. Override `getTestDataPath()` - default points to IntelliJ Platform sources
3. Enable inspections explicitly: `fixture.enableInspections(MyInspection::class.java)`
4. For Java PSI tests: Set `idea.home.path` system property to IntelliJ Community sources

### Integration Tests

**Requirements:**
- **JUnit 5 only** (uses JUnit 5 extensions)
- Separate source set: `src/integrationTest/kotlin`
- Built plugin ZIP distribution
- Use **identical versions** for all framework libraries

```kotlin
val integrationTestSourceSet = sourceSets.create("integrationTest") {
    java.srcDir("src/integrationTest/kotlin")
    resources.srcDir("src/integrationTest/resources")
}

configurations {
    named("integrationTestImplementation") {
        // DON'T extend from test configurations (framework conflicts)
    }
}

dependencies {
    intellijPlatform {
        testFramework(TestFrameworkType.Starter)
    }
    add("integrationTestImplementation", sourceSets.main.get().output)
    "integrationTestImplementation"("org.junit.jupiter:junit-jupiter-api:5.10.2")
    "integrationTestRuntimeOnly"("org.junit.jupiter:junit-jupiter-engine:5.10.2")
    "integrationTestImplementation"(kotlin("test"))
    "integrationTestImplementation"("org.kodein.di:kodein-di-jvm:7.20.2")
    "integrationTestImplementation"("org.jetbrains.kotlinx:kotlinx-coroutines-core-jvm:1.7.3")
}

val integrationTest = register<Test>("integrationTest") {
    testClassesDirs = integrationTestSourceSet.output.classesDirs
    classpath = integrationTestSourceSet.runtimeClasspath
    useJUnitPlatform()

    val buildPlugin = named<Zip>("buildPlugin")
    dependsOn(buildPlugin)

    doFirst {
        systemProperty("path.to.build.plugin",
            buildPlugin.flatMap { it.archiveFile }.get().asFile.absolutePath)
    }
}
```

---

## Common Test Patterns

### Test Helpers

| Method | Purpose |
|--------|---------|
| `type(text)` | Simulate keyboard input |
| `performEditorAction(id)` | Trigger editor actions |
| `complete()` | Code completion (returns lookup items) |
| `findUsages()` | Find Usages simulation |
| `renameElementAtCaret(name)` | Rename refactoring |
| `findSingleIntention(text)` + `launchAction()` | Execute intention/quick fix |
| `checkResultByFile(expected)` | Compare output |

### File Setup

```kotlin
myFixture.configureByFile("Main.kt")              // Open in editor
myFixture.configureByFiles("Main.kt", "Util.kt")  // Multiple files
myFixture.copyFileToProject("src.kt", "dst.kt")   // Copy without opening
```

### Test Data Organization

```
src/
├── test/
│   ├── kotlin/          # Test code
│   └── testData/        # Test files (NOT under source root!)
│       └── myfeature/
│           ├── input.kt
│           └── input_after.kt
└── integrationTest/kotlin/
```

**Naming convention:**
```kotlin
fun testRenameVariable() {
    val name = getTestName(false)  // "renameVariable"
    myFixture.configureByFile("$name.kt")
    // ...
    myFixture.checkResultByFile("${name}_after.kt")
}
```

**Special markup:**
```kotlin
val x<caret> = 42                              // Caret position
val text = "<selection>selected</selection>"   // Selection
<block>col1 col2\ncol1 col2</block>             // Block selection
```

---

## Testing Specific Features

### Highlighting & Inspections

```kotlin
fun testHighlighting() {
    myFixture.enableInspections(MyInspection::class.java)
    myFixture.configureByFile("test.kt")
    myFixture.checkHighlighting()
}
```

**Expected results in test file:**
```kotlin
<warning descr="Variable is never used">val unused = 42</warning>
<error descr="Unresolved reference">unknownFunction</error>()
```

**Severity tags:** `<error>`, `<warning>`, `<weak_warning>`, `<info>`, `<inject>`, `<symbolName>`

**Tag attributes:** `descr`, `tooltip`, `textAttributesKey`, `effectType`, `fontType`

### Code Completion

```kotlin
fun testCompletion() {
    myFixture.configureByFile("test.kt")
    val variants = myFixture.complete()
    assertTrue(variants.any { it.lookupString == "expectedItem" })
}
```

### Refactorings

```kotlin
fun testRename() {
    myFixture.configureByFile("before.kt")
    myFixture.renameElementAtCaret("newName")
    myFixture.checkResultByFile("after.kt")
}
```

---

## Integration Tests

**Architecture:** Dual-process (test process + IDE process in separate JVM)
**Communication:** JMX/RMI for API, Driver framework for UI
**Note:** First run downloads IDE; subsequent runs use cache

### Basic Test Structure

```kotlin
@Test
fun testPluginFunctionality() = runBlocking {
    Starter.newContext {
        ide(IdeDownloader.ideFromSources(), useLatestDownloadedIdeBuild = false)
        idea {
            build = IjSdkVersion("2024.2")
            configDir = testProjectDir.resolve("config")
            systemDir = testProjectDir.resolve("system")
        }
        withDriver {
            driver(DriverConfig(testProjectDir.resolve("driver")))
        }
        project {
            fromGithub("username/repo", "main")
        }
        plugins {
            plugin(PluginConfigurator.fromLocalBuild())
        }
    }.runIdeWithDriver().useDriverAndCloseIde { driver, ide ->
        driver.waitForIndicators()
        // Test code here
    }
}
```

### UI Testing

**Component finding priority:**
1. `byAccessibleName()` - Most reliable
2. `byVisibleText()` - Displayed text
3. `byType()` - Java class
4. `byAttribute("name", "value")` - Custom attributes

```kotlin
driver.ideFrame {
    codeEditor {
        waitText("content", timeout = Duration.ofSeconds(30))
        insertText("new text")
    }

    button {
        xQuery { byAccessibleName("OK") }
    }.click()

    tree {
        xQuery { byType<JTree>() }
    }.shouldBe { rawItems.contains("MyFile.kt") }
}
```

**Debug UI:** While test paused, visit `http://localhost:63343/api/remote-driver/` to inspect component tree.

### API Testing with Remote Stubs

```kotlin
@Remote("com.example.MyService", plugin = "com.example.plugin")
interface MyServiceStub {
    fun doSomething(param: String): String
}

// In test:
val service = ide.service<MyServiceStub>()
val result = service.doSomething("test")
```

**RMI limitations:** Primitives, Strings, `@Remote` refs, arrays, lists only; public methods only; no suspend functions.

### Exception Handling

IDE exceptions don't auto-propagate. Use Kodein DI to configure:

```kotlin
init {
    di = DI {
        extend(di)
        bindSingleton<CIServer>(overrides = true) {
            object : CIServer by NoCIServer {
                override fun reportTestFailure(
                    testName: String, message: String, details: String
                ) {
                    fail { "$testName fails: $message" }
                }
            }
        }
    }
}
```

---

## Best Practices

### Essential Rules

1. **tearDown in finally block:**
   ```kotlin
   @After
   fun tearDown() {
       try { /* cleanup */ }
       finally { super.tearDown() }  // CRITICAL!
   }
   ```

2. **Avoid flaky tests:**
   - No OS-specific assumptions
   - Use `assertUnorderedCollection()` for unordered data
   - Wait for conditions, never `Thread.sleep()`

3. **Wait properly:**
   ```kotlin
   // Bad: Thread.sleep(1000)
   // Good:
   myFixture.waitForCondition(timeout = 5000) { someCondition() }
   ```

4. **Indexing (2024.2+):**
   ```kotlin
   IndexingTestUtil.waitUntilIndexesAreReady(project)  // Async indexing!
   ```

5. **Debug logging:**
   ```
   -Didea.log.debug.categories=#com.example.plugin
   -Didea.split.test.logs=true
   ```

6. **VFS refresh issues:** Delete `test-system/caches` in sandbox if files not visible

---

## Useful Utilities

| Class | Purpose |
|-------|---------|
| `UsefulTestCase` | General assertions, test utilities |
| `PlatformTestUtil` | Directory comparison, platform helpers |
| `CodeInsightTestUtil` | Code insight testing |
| `PsiTestUtil` | PSI manipulation, library/source root management |
| `IndexingTestUtil` | Index waiting (2024.2+) |
| `ExtensionTestUtil` | Replace extensions in tests |
| `EditorTestUtil` | Editor-specific helpers |

### Common Patterns

```kotlin
// Replace service
project.replaceService(MyService::class.java, testService, testRootDisposable)

// Replace extension point
ExtensionTestUtil.maskExtensions(extensionPoint, listOf(testExt), testRootDisposable)

// Add test library
PsiTestUtil.addLibrary(module, "mylib", "/path/to/lib", "lib.jar")
```

---

## Quick Decision Tree

```
UI interactions needed?
├─ YES → Integration Test with Driver
└─ NO → Continue...
    Multiple modules?
    ├─ YES → Heavy Test
    └─ NO → Light Test
        Java PSI needed?
        ├─ YES → LightJavaCodeInsightFixtureTestCase
        └─ NO → BasePlatformTestCase
```

---

## Summary

**Default approach:**
- Light tests for speed
- Test data in dedicated directory (not source root)
- Name test methods to match test files
- Enable inspections explicitly
- Direct model testing over UI
- Integration tests only for UI/complex scenarios
- Always use `finally` for tearDown
- Wait for conditions, never sleep
- Use real implementations, avoid mocking

**The framework prioritizes stability** - write tests that focus on feature behavior and leverage provided utilities.
