# Clone Progress Improvements

## Current State

The current clone implementation uses `runProcessWithProgressSynchronously` which only shows a static "Cloning Repository..." message. Users don't see real-time progress during potentially long clone operations.

## Research Findings from IntelliJ Platform

### Git Clone Progress Architecture

The Git plugin uses sophisticated multi-layered progress indication:

**Core Components:**
- **GitStandardProgressAnalyzer**: Parses Git's standard output to extract progress information
- **GitLineHandler**: Streams command output line-by-line in real-time
- **BufferingTextSplitter**: Handles proper line splitting, including CR-only progress lines
- **CloneableProjectsService**: Manages the overall clone task lifecycle

**Progress Analysis Pattern:**
```java
// GitStandardProgressAnalyzer recognizes these Git operations:
COUNTING_OBJECTS(".*Counting objects: +(\\d+).*", 0.05)     // 5% of total
COMPRESSING_OBJECTS(".*Compressing objects: +(\\d{1,3})%.*", 0.1)  // 10% of total  
RECEIVING_OR_WRITING_OBJECTS(".*(?:Receiving|Writing) objects: +(\\d{1,3})%.*", 0.8) // 80% of total
RESOLVING_DELTAS(".*Resolving deltas: +(\\d{1,3})%.*", 0.05)  // 5% of total
```

**Real-time Progress Updates:**
```java
GitLineHandlerListener progressListener = GitStandardProgressAnalyzer.createListener(indicator);
// This listener parses each line and updates:
indicator.setIndeterminate(false);
indicator.setFraction(fraction);  // 0.0 to 1.0 based on operation stage
indicator.setText2(line);         // Shows current Git output line
```

### SVN Clone Progress Architecture

**Event-Driven Progress:**
- **ProgressTracker**: Interface for handling SVN progress events
- **UpdateEventHandler**: Converts SVN events to progress updates
- **CheckoutEventHandler**: Specialized for checkout operations

**Detailed Progress Information:**
```java
// SVN provides granular progress events:
UPDATE_ADD, UPDATE_DELETE, UPDATE_UPDATE, UPDATE_EXTERNAL, UPDATE_COMPLETED
// Each event includes file path, action type, and revision information
```

### Mercurial (Hg) Approach

Hg uses a simpler but effective streaming approach:
```java
// Shows each output line as progress details
if (ProcessOutputTypes.STDOUT == outputType && indicator != null && showTextOnIndicator) {
    indicator.setText2(line);  // Real-time command output display
}
```

### Available ProgressIndicator APIs

**Basic Progress Methods:**
- `setText(String)` - Main progress description
- `setText2(String)` - Detailed progress information (sub-text)
- `setFraction(double)` - Progress bar (0.0 to 1.0)
- `setIndeterminate(boolean)` - Indeterminate vs determinate progress

**Advanced Features:**
- `pushState()` / `popState()` - Save/restore progress state for nested operations
- `checkCanceled()` - Handle user cancellation
- Progress indicator hierarchies and delegation

### Streaming Process Output Patterns

**Real-time Line Processing:**
```java
// BufferingTextSplitter handles:
// - CR-only lines (progress updates that overwrite)
// - CRLF line endings
// - Partial line buffering
// - Real-time streaming without blocking

interface LineConsumer {
    void consume(String line, boolean isCrOnly);  // isCrOnly indicates progress line
}
```

**Output Parsing Strategies:**
1. **Regex-based parsing** (Git) - Extract percentages and operation types
2. **Event conversion** (SVN) - Convert command output to structured events  
3. **Direct streaming** (Hg) - Show raw output with optional filtering

### Clone Task Framework

**CloneableProjectsService Pattern:**
```java
CloneTask cloneTask = new CloneTask() {
    public CloneTaskInfo taskInfo() { /* task metadata */ }
    public CloneStatus run(ProgressIndicator indicator) { 
        // Execute clone with progress updates
        // Return SUCCESS/FAILURE/CANCEL
    }
};
CloneableProjectsService.getInstance().runCloneTask(projectPath, cloneTask);
```

## Recommendations for Better JJ Clone Progress

### 1. Implement a JJ Progress Analyzer
```kotlin
class JjProgressAnalyzer {
    // Parse JJ clone output for progress indicators
    // Look for patterns like file counts, transfer progress, etc.
    fun analyzeProgress(line: String): Double
}
```

### 2. Use Real-time Streaming
```kotlin
// Follow GitLineHandler pattern:
val handler = JjLineHandler(project, workingDir, JjCommand.CLONE)
handler.addLineListener(JjProgressAnalyzer.createListener(indicator))
handler.addParameters(url, destination)
```

### 3. Leverage Platform Clone Framework
```kotlin
// Use CloneableProjectsService for consistent clone UI:
val cloneTask = JjCloneTask(url, directory)
CloneableProjectsService.getInstance().runCloneTask(projectPath, cloneTask)
```

### 4. Handle Different Progress Phases
```kotlin
// JJ clone might have phases like:
// - Repository discovery
// - Object transfer  
// - Working copy checkout
// - Post-clone processing
// Weight each phase appropriately in total progress
```

### 5. Provide Meaningful Progress Text
```kotlin
indicator.setText("Cloning JJ repository...")
indicator.setText2("Downloading objects: 1247/2340 files") // Real-time details
indicator.setFraction(0.53) // Based on actual progress analysis
```

## Implementation Strategy

Since `jj git clone` uses Git's progress output format, we can adapt Git's progress parsing patterns:

1. **Real-time streaming with `ProgressIndicator`** - Parse `jj git clone` output line-by-line
2. **Show current operation** (e.g., "Receiving objects: 45%")
3. **Display file counts and transfer progress**
4. **Provide determinate progress bar** based on Git's percentage output

## Benefits

- Users see real-time progress during long clone operations
- Determinate progress bars instead of indeterminate spinners
- Detailed progress text showing current operation
- Consistent with other IntelliJ VCS clone experiences
- Ability to handle user cancellation gracefully

## Next Steps

1. Create `JjLineHandler` and `JjProgressAnalyzer` classes
2. Implement real-time output streaming
3. Parse Git progress patterns from `jj git clone` output
4. Replace current `runProcessWithProgressSynchronously` with streaming approach
5. Test with various repository sizes and network conditions