# JJ Plugin Implementation Plan

## Refined Architecture Plan - Repository Integration with LogProvider

Based on analysis of git4idea and hg4idea implementations in IntelliJ Platform sources.

### 1. Core Components to Create

**SelvejjRepository Interface & Implementation:**

- Implement `com.intellij.dvcs.repo.Repository` interface
- Do the appropriate setup of SelvejjRepository
- stub out all overrides just enough to get the compiler happy.

**SelvejjRepositoryCreator:**

- Implement `VcsRepositoryCreator` interface
- Call it SelvejjRepositoryCreator - again just stub out things.
- Create `SelvejjRepository` instances for valid jj repos
- Add the necessary extension point in plugin.xml to hook this class up.

### 2. File System Monitoring & Updates

**SelvejjRepositoryUpdater (Key Component):**

- Implement `BulkFileListener` and `Disposable`
- Watch `.jj/working_copy/checkout` for changes
- Use `LocalFileSystem.addRootToWatch()` with `MergingUpdateQueue`
- Pattern: File change → Queue update → Call `repository.update()` → Publish message

**Message Bus Integration:**

```java
// Define topic in SelvejjRepository
Topic<SelvejjRepositoryChangeListener> JJ_REPO_CHANGE =
        Topic.create("SelvejjRepository change", SelvejjRepositoryChangeListener.class);

// In updater/manager - publish changes  
syncPublisher(project, JJ_REPO_CHANGE).

repositoryChanged(repository);
```

### 3. LogProvider Integration

**Update LogProvider:**

- Subscribe to `JJ_REPO_CHANGE` topic in constructor/init
- Add refresh logic: `connection.subscribe(JJ_REPO_CHANGE, repo -> refresher.refresh(repo.getRoot()))`

### 5. Implementation Pattern

1. **File change detected** → `SelvejjRepositoryUpdater.after(events)`
2. **Queue repository update** → `MergingUpdateQueue` (debounced)
3. **Update repository state** → `SelvejjRepository.update()` reads from disk
4. **Publish message** → `JJ_REPO_CHANGE` topic notification
5. **LogProvider refreshes** → Subscribers receive change event and refresh UI

---

## Legacy Notes - Potentially useful classes for filesystem/VFS operations

* com/intellij/openapi/vfs/VfsUtil.java
* com/intellij/openapi/vfs/AsyncFileListener.java
* com/intellij/vfs/AsyncVfsEventsListener.java
* com/intellij/openapi/progress/util/BackgroundTaskUtil.java

## Legacy Analysis

We will need some mechanism (VFS watches on `.jj`, mostly just on
`.jj/working_copy/checkout`) https://github.com/jj-vcs/jj/discussions/6646
to observe changes that should lead to refreshing state.

At a minimum, the log provider should be cognizant of this as it epic fails right now.
This monitor likely needs to be per-project or per-root?
One option is to leverage the `Repository`, `RepositoryCreator` interfaces from the IJ platform (along with some of the
neighboring abstract classes that may provide utilities)

The hg/git impls seem to do this, then have a RepositoryUpdater that is set up in the Disposable heirarchy. This
monitors relevant files, and then updates the repo, which then publishes a message on a new topic. The log provider and
others seem to listen to it.