/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.execution.util.ExecUtil
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.project.Project
import com.intellij.openapi.util.NlsContexts
import com.intellij.openapi.util.NlsSafe
import com.intellij.openapi.util.io.FileUtil
import com.intellij.openapi.vcs.FilePath
import com.intellij.openapi.vcs.LocalFilePath
import com.intellij.openapi.vcs.RepositoryLocation
import com.intellij.openapi.vcs.VcsException
import com.intellij.openapi.vcs.annotate.FileAnnotation
import com.intellij.openapi.vcs.annotate.LineAnnotationAspect
import com.intellij.openapi.vcs.annotate.LineAnnotationAspectAdapter
import com.intellij.openapi.vcs.history.VcsFileRevision
import com.intellij.openapi.vcs.history.VcsFileRevisionEx
import com.intellij.openapi.vcs.history.VcsRevisionNumber
import com.intellij.openapi.vcs.vfs.VcsFileSystem
import com.intellij.openapi.vcs.vfs.VcsVirtualFile
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.vcs.AnnotationProviderEx
import com.intellij.vcsUtil.VcsUtil
import kotlinx.datetime.toJavaInstant
import kotlinx.serialization.json.Json
import org.jetbrains.annotations.NonNls
import java.util.*

class SelvejjFileRevision(
    private val details: SelvejjCommitDetails,
    private val vcsRoot: FilePath,
    private val path: FilePath
) : VcsFileRevisionEx() {
    companion object {
        private val LOG = logger<SelvejjFileRevision>()
    }

    override fun getAuthorEmail(): @NlsSafe String? {
        return details.author.email
    }

    override fun getCommitterName(): @NlsSafe String? {
        return details.committer.name
    }

    override fun getCommitterEmail(): @NlsSafe String? {
        return details.committer.email
    }

    override fun getPath(): FilePath {
        return path
    }

    override fun getAuthorDate(): Date? {
        return Date.from(details.author.timestamp.toJavaInstant())
    }

    override fun isDeleted(): Boolean {
        TODO("Not yet implemented")
    }

    override fun getBranchName(): @NlsSafe String? {
        TODO("Not yet implemented")
    }

    override fun getChangedRepositoryPath(): RepositoryLocation? {
        TODO("Not yet implemented")
    }

    override fun loadContent(): ByteArray? {
        val commitId = details.commit_id
        val rootPath = vcsRoot.path
        val absolutePath = path.path
        val relativePath = FileUtil.getRelativePath(rootPath, absolutePath, '/') ?: absolutePath

        val command = GeneralCommandLine("jj")
            .withParameters("file", "show", "-r", commitId, relativePath)
            .withWorkDirectory(rootPath)

        // TODO: Consider switching to an execution mode that gives direct access to the bytes instead of converting to
        //  string and then forcing us to convert back.
        val result = ExecUtil.execAndGetOutput(command, 10000)
        return if (result.checkSuccess(LOG)) result.stdout.toByteArray(Charsets.UTF_8) else {
            LOG.warn("Failed to load content for $relativePath at $commitId: ${result.stderr}")
            null
        }
    }

    @Deprecated("Deprecated in Java")
    override fun getContent(): ByteArray? {
        TODO("Not yet implemented")
    }

    override fun getRevisionNumber(): VcsRevisionNumber {
        return SelvejjRevisionNumber(details.commit_id)
    }

    override fun getRevisionDate(): Date? {
        return Date.from(details.committer.timestamp.toJavaInstant())
    }

    override fun getAuthor(): @NlsSafe String? {
        return details.author.name
    }

    override fun getCommitMessage(): @NlsSafe String? {
        return details.description
    }
}

// TODO: Provide richer tooltips
// test across renames (i.e. it should go to the correct file)
class SelvejjAnnotation(
    private val project: Project,
    private val file: VirtualFile,
    private val fileAnnotations: List<SelvejjCommitDetails>,
    private val revision: SelvejjRevisionNumber,
    private val fileRevisions: List<VcsFileRevision>
) : FileAnnotation(project) {
    companion object {
        private val LOG = logger<SelvejjAnnotation>()
        private const val CHANGE_ID = "ChangeId"
    }

    override fun getFile(): VirtualFile {
        return file
    }

    inner class SelvejjLineAnnotationAspect(
        id: String,
        displayName: String,
        showByDefault: Boolean
    ) : LineAnnotationAspectAdapter(id, displayName, showByDefault) {
        override fun showAffectedPaths(p0: Int) {
            TODO("Not yet implemented")
        }

        override fun getValue(lineNumber: Int): @NlsSafe String? {
            if (lineNumber < 0 || lineNumber >= fileAnnotations.size) return null
            return when (id) {
                LineAnnotationAspect.AUTHOR -> fileAnnotations[lineNumber].author.name
                LineAnnotationAspect.DATE -> fileAnnotations[lineNumber].author.timestamp.toString()
                LineAnnotationAspect.REVISION -> fileAnnotations[lineNumber].commit_id
                CHANGE_ID -> fileAnnotations[lineNumber].change_id
                else -> null
            }
        }
    }

    override fun getAnnotatedContent(): @NonNls String? {
        val commitId = revision.getCommitId()
        val vcsRoot = VcsUtil.getVcsRootFor(project, VcsUtil.getFilePath(file.path, false))!!
        // having to do this because VfsUtilCore relative computation somehow doesn't work for these VirtualFiles coming from the annotation action.
        val relativePath = FileUtil.getRelativePath(vcsRoot.path, file.path, '/') ?: file.path

        val command = GeneralCommandLine("jj")
            .withParameters("file", "show", "-r", commitId, relativePath!!)
            .withWorkDirectory(vcsRoot.path)

        // TODO: Consider switching to an execution mode that gives direct access to the bytes instead of converting to
        //  string and then forcing us to convert back.
        val result = ExecUtil.execAndGetOutput(command, 10000)
        return if (result.checkSuccess(LOG)) result.stdout else {
            LOG.warn("Failed to load content for $relativePath at $commitId: ${result.stderr}")
            null
        }
    }

    override fun getCurrentRevision(): VcsRevisionNumber? {
        return revision
    }

    override fun getAspects(): Array<out LineAnnotationAspect?> {
        return arrayOf(
            SelvejjLineAnnotationAspect(LineAnnotationAspect.AUTHOR, "Author", true),
            SelvejjLineAnnotationAspect(LineAnnotationAspect.DATE, "Date", true),
            SelvejjLineAnnotationAspect(CHANGE_ID, "Change ID", true),
            SelvejjLineAnnotationAspect(LineAnnotationAspect.REVISION, "Revision", false)
        )
    }

    override fun getLineCount(): Int {
        return fileAnnotations.size
    }

    override fun getToolTip(lineNumber: Int): @NlsContexts.Tooltip String? {
        return null
    }

    override fun getLineRevisionNumber(lineNumber: Int): VcsRevisionNumber? {
        if (lineNumber < 0 || lineNumber >= fileAnnotations.size) return null
        return SelvejjRevisionNumber(fileAnnotations[lineNumber].commit_id)
    }

    override fun getLineDate(lineNumber: Int): Date? {
        if (lineNumber < 0 || lineNumber >= fileAnnotations.size) return null
        return Date.from(fileAnnotations[lineNumber].committer.timestamp.toJavaInstant())
    }

    override fun getRevisions(): List<VcsFileRevision?>? {
        return fileRevisions
    }

    fun getChangeId(lineNumber: Int): String? {
        return if (lineNumber >= 0 && lineNumber < fileAnnotations.size) {
            fileAnnotations[lineNumber].change_id
        } else null
    }
}


class SelvejjAnnotationProvider(private val project: Project) : AnnotationProviderEx {

    companion object {
        private val LOG = logger<SelvejjAnnotationProvider>()
        private var ANNOTATION_TEMPLATE = "json(commit)"
    }

    override fun annotate(
        filePath: FilePath,
        revNum: VcsRevisionNumber
    ): FileAnnotation {
        val vcsRoot = VcsUtil.getVcsRootFor(project, filePath)
            ?: throw VcsException("Could not find VCS root for ${filePath.path}")

        val fileRevision = SelvejjFileRevision(
            loadCommitDetails(vcsRoot, revNum as SelvejjRevisionNumber),
            LocalFilePath(vcsRoot.path, true),
            filePath
        )

        // Create a VcsVirtualFile using the string path constructor
        val file = VcsVirtualFile(filePath.path, fileRevision, VcsFileSystem.getInstance())

        return SelvejjAnnotation(
            project,
            file,
            buildAnnotationRevisions(vcsRoot, file, revNum),
            revNum,
            buildFileHistoryRevisions(vcsRoot, filePath)
        )
    }

    override fun annotate(file: VirtualFile): FileAnnotation {
        val vcsRoot = getRepositoryRoot(file)!!;
        val commitId = runJjCommand(vcsRoot.path, "log", "-r", "@", "-T", "commit_id", "--no-graph")
        val filePath = VcsUtil.getFilePath(file)
        return SelvejjAnnotation(
            project,
            file,
            buildAnnotationRevisions(vcsRoot, file, SelvejjRevisionNumber(commitId!!)),
            SelvejjRevisionNumber(commitId!!),
            buildFileHistoryRevisions(vcsRoot, filePath)
        )
    }

    override fun annotate(
        file: VirtualFile,
        revision: VcsFileRevision
    ): FileAnnotation {
        val vcsRoot = getRepositoryRoot(file)!!;
        val filePath = if (revision is VcsFileRevisionEx) revision.path else VcsUtil.getFilePath(file)
        return SelvejjAnnotation(
            project,
            file,
            buildAnnotationRevisions(vcsRoot, file, revision.revisionNumber as SelvejjRevisionNumber),
            revision.revisionNumber as SelvejjRevisionNumber,
            buildFileHistoryRevisions(vcsRoot, filePath)
        )
    }

    private fun getRepositoryRoot(file: VirtualFile): VirtualFile? {
        return VcsUtil.getVcsRootFor(project, VcsUtil.getFilePath(file.path, false))
    }

    private fun runJjCommand(workingDir: String, vararg args: String): String? {
        val command = GeneralCommandLine("jj")
            .withParameters(args.toList())
            .withWorkDirectory(workingDir)

        val result = ExecUtil.execAndGetOutput(command, 10000)
        return if (result.checkSuccess(LOG)) result.stdout.trim() else null
    }

    private fun loadCommitDetails(vcsRoot: VirtualFile, revNum: SelvejjRevisionNumber): SelvejjCommitDetails {
        val command = GeneralCommandLine("jj")
            .withParameters(
                "log",
                "-R", vcsRoot.path,
                "--no-graph",
                "-r", revNum.getCommitId(),
                "-T", "json(self)"
            )

        val result = ExecUtil.execAndGetOutput(command, 10000)
        if (!result.checkSuccess(LOG)) {
            throw VcsException("Failed to load commit details for ${revNum.getCommitId()}: ${result.stderr}")
        }

        val json = Json { ignoreUnknownKeys = true }
        return json.decodeFromString<SelvejjCommitDetails>(result.stdout.trim())
    }

    private fun buildAnnotationRevisions(
        vcsRoot: VirtualFile,
        file: VirtualFile,
        revNum: SelvejjRevisionNumber
    ): List<SelvejjCommitDetails> {
        return loadAnnotationDetails(vcsRoot, file, revNum.getCommitId(), ANNOTATION_TEMPLATE)
    }

    // Builds a chronologically ordered (newest -> oldest) list of file-level revisions for the given file.
    // This uses `jj log --no-graph` filtered by the file path, and deserializes each line of JSON into
    // SelvejjCommitDetails, then wraps each into a SelvejjFileRevision.
    private fun buildFileHistoryRevisions(
        vcsRoot: VirtualFile,
        filePath: FilePath
    ): List<VcsFileRevision> {
        val relativePath = FileUtil.getRelativePath(vcsRoot.path, filePath.path, '/') ?: return emptyList()

        val command = GeneralCommandLine("jj")
            .withParameters(
                "log",
                "-R", vcsRoot.path,
                "--no-graph",
                "-r", "all()",
                "-T", "json(self) ++ \"\\n\"",
                "root:'${relativePath}'"
            )

        val result = ExecUtil.execAndGetOutput(command, 30000)
        if (!result.checkSuccess(LOG)) {
            throw RuntimeException("jj log command failed: ${result.stderr}")
        }

        val json = Json { ignoreUnknownKeys = true }
        return result.stdout.lines()
            .filter { it.trim().isNotEmpty() }
            .mapNotNull { line ->
                try {
                    val details = json.decodeFromString<SelvejjCommitDetails>(line.trim())
                    // TODO: Support renames by passing the path at the revision.
                    // To do so, need to run log with a more complicated expression, where we ask each commit for the
                    // list of files changed, and check if the status of any of them is a rename. Then, store the source
                    // path (the old name).
                    // jj log will currently stop the file history at the rename commit. To get the full history,
                    // additional command executions are also required.
                    SelvejjFileRevision(
                        details,
                        LocalFilePath(vcsRoot.path, true),
                        filePath
                    ) as VcsFileRevision
                } catch (e: Exception) {
                    LOG.warn("Failed to parse file history revision line: $line", e)
                    null
                }
            }
    }

    private fun loadAnnotationDetails(
        root: VirtualFile,
        file: VirtualFile,
        commitId: String,
        template: String
    ): List<SelvejjCommitDetails> {
        val command = GeneralCommandLine("jj")
            .withParameters(
                "file",
                "annotate",
                "-R",
                root.path,
                "-r",
                commitId,
                "-T",
                "$template ++ \"\\n\"",
                file.path
            )

        val result = ExecUtil.execAndGetOutput(command, 30000)
        if (!result.checkSuccess(LOG)) {
            throw RuntimeException("jj command failed: ${result.stderr}")
        }

        val json = Json { ignoreUnknownKeys = true }
        return result.stdout.lines()
            .filter { it.trim().isNotEmpty() }
            .mapNotNull { line ->
                try {
                    json.decodeFromString<SelvejjCommitDetails>(line.trim())
                } catch (e: Exception) {
                    LOG.warn("Failed to parse commit details line: $line", e)
                    null
                }
            }
    }
}
