/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.diff.util.Side
import com.intellij.dvcs.repo.VcsRepositoryManager
import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.openapi.application.ApplicationManager
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.project.Project
import com.intellij.openapi.util.io.FileUtil
import com.intellij.openapi.vcs.FilePath
import com.intellij.openapi.vcs.VcsException
import com.intellij.openapi.vcs.changes.Change
import com.intellij.openapi.vcs.changes.ChangeListChange
import com.intellij.openapi.vcs.changes.CommitContext
import com.intellij.openapi.vcs.checkin.CheckinEnvironment
import com.intellij.openapi.vcs.ex.PartialCommitHelper
import com.intellij.openapi.vcs.ex.PartialLocalLineStatusTracker
import com.intellij.openapi.vcs.impl.PartialChangesUtil
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.vcsUtil.VcsUtil
import kotlin.io.path.Path
import kotlin.io.path.listDirectoryEntries

class SelvejjCheckinEnvironment(private val project: Project) : CheckinEnvironment {

    companion object {
        private val LOG = logger<SelvejjCheckinEnvironment>()
    }

    override fun getCheckinOperationName(): String = "Commit"

    override fun getHelpId(): String? = null

    /**
     * Data class to hold partial change information with transaction helper
     */
    private data class PartialFileChange(
        val change: Change,
        val virtualFile: VirtualFile,
        val helper: PartialCommitHelper
    )

    /**
     * Data class to hold full change information
     */
    private data class FullFileChange(
        val change: Change,
        val virtualFile: VirtualFile?
    )

    override fun commit(
        changes: List<Change>,
        commitMessage: String,
        commitContext: CommitContext,
        feedback: MutableSet<in String>
    ): List<VcsException>? {
        val exceptions = mutableListOf<VcsException>()

        if (changes.isEmpty()) {
            exceptions.add(VcsException("Cannot commit with no changes"))
            return exceptions
        }

        try {
            // Separate partial and full changes following Git's pattern
            val (partialChanges, fullChanges) = separatePartialAndFullChanges(changes)

            LOG.info("Found ${partialChanges.size} partial changes and ${fullChanges.size} full changes")

            // Group both types of changes by repository root
            val changesByRoot = groupChangesByRoot(partialChanges, fullChanges)
            var successfulRoots = 0
            val totalRoots = changesByRoot.size

            for (rootChanges in changesByRoot) {
                try {
                    performCommitForRoot(rootChanges, commitMessage, feedback)
                    successfulRoots++
                } catch (e: VcsException) {
                    exceptions.add(e)
                    LOG.warn("Failed to commit in root ${rootChanges.root.path}", e)
                }
            }

            // Add summary feedback
            if (successfulRoots > 0) {
                if (successfulRoots == totalRoots) {
                    feedback.add("All repositories committed successfully")
                } else {
                    feedback.add("$successfulRoots of $totalRoots repositories committed successfully")
                }
            }

        } catch (e: Exception) {
            LOG.warn("Failed to commit changes", e)
            exceptions.add(VcsException("Commit operation failed: ${e.message}", e))
        }

        return if (exceptions.isNotEmpty()) exceptions else null
    }

    /**
     * Data class to hold both partial and full changes for a repository root
     */
    private data class RootChanges(
        val root: VirtualFile,
        val partialChanges: List<PartialFileChange>,
        val fullChanges: List<FullFileChange>
    )

    /**
     * Group both partial and full changes by repository root
     */
    private fun groupChangesByRoot(
        partialChanges: List<PartialFileChange>,
        fullChanges: List<FullFileChange>
    ): List<RootChanges> {
        val rootMap = mutableMapOf<VirtualFile, MutableList<PartialFileChange>>()
        val rootFullMap = mutableMapOf<VirtualFile, MutableList<FullFileChange>>()

        // Group partial changes by root
        for (partialChange in partialChanges) {
            val root = VcsUtil.getVcsRootFor(project, partialChange.virtualFile)
            if (root != null) {
                rootMap.getOrPut(root) { mutableListOf() }.add(partialChange)
            }
        }

        // Group full changes by root
        for (fullChange in fullChanges) {
            val filePath = fullChange.change.afterRevision?.file ?: fullChange.change.beforeRevision?.file
            if (filePath != null) {
                val root = VcsUtil.getVcsRootFor(project, filePath)
                if (root != null) {
                    rootFullMap.getOrPut(root) { mutableListOf() }.add(fullChange)
                }
            }
        }

        // Combine both maps to create RootChanges
        val allRoots = (rootMap.keys + rootFullMap.keys).distinct()
        return allRoots.map { root ->
            RootChanges(
                root = root,
                partialChanges = rootMap[root] ?: emptyList(),
                fullChanges = rootFullMap[root] ?: emptyList()
            )
        }
    }

    private fun performCommitForRoot(
        rootChanges: RootChanges,
        commitMessage: String,
        feedback: MutableSet<in String>
    ) {
        val root = rootChanges.root
        val partialChanges = rootChanges.partialChanges
        val fullChanges = rootChanges.fullChanges

        if (partialChanges.isEmpty() && fullChanges.isEmpty()) {
            return // Nothing to commit for this root
        }

        LOG.info("Committing to ${root.path}: ${partialChanges.size} partial + ${fullChanges.size} full changes")

        FakeDiffEditor().use { editor ->
            // Build jj commit command with specific files
            val command = GeneralCommandLine("jj")
                .withParameters("-R", root.path, "commit", "--tool", editor.scriptPath, "--message", commitMessage)
                .withWorkDirectory(root.path)

            // Start jj commit in background and communicate with script
            val (leftPath, rightPath) = editor.startAndWaitForReady(command)
            LOG.debug("Received paths from script: left=${leftPath}, right=${rightPath}")

            syncChangesToSparseCheckout(rootChanges, leftPath, rightPath)

            // Signal script to exit with success
            editor.setExitCode(0)

            // The close() method will handle waiting for the process to complete
            // If it succeeds, provide feedback; otherwise an exception will be thrown
            feedback.add("Successfully committed changes in ${root.name}")

            // Apply partial changes on EDT
            if (partialChanges.isNotEmpty()) {
                ApplicationManager.getApplication().invokeLater {
                    partialChanges.forEach {
                        try {
                            it.helper.applyChanges()
                        } catch (e: Exception) {
                            LOG.warn("Failed to apply partial changes for ${it.virtualFile.path}", e)
                        }
                    }
                }
            }

            refreshRepository(root)
        }
        // Seems like stuff works on the jj side, but then we hit this exception
        // Commit operation failed: Cannot invoke (class=MyCommandListener, method=undoTransparentActionStarted, topic=CommandListener) (see idea.log as well)
        // probably a EDT thing.
    }

    private fun syncChangesToSparseCheckout(
        rootChanges: RootChanges,
        leftPath: String,
        rightPath: String
    ) {
        // When jj commit --tool foo is invoked, the relevant binary for foo is called with left and right directories.
        // We do need to make sure to preserve the mode of the file from the working copy, which I guess should happen automatically via jj.
        // So I think we should really only change the contents where it makes sense.
        // For "full files", we should copy the entire content over.
        // However, we need to be careful to remove the files that aren't getting added.
        // I think this is where making the right match the left first may be the way to go.
        // if a file is deleted in the working copy, but then not selected for the commit, it should be added back to right.
        // if a file is deleted in the working copy, and selected for commit, right should be kept as is.
        // ok, so for deletions, it will be in the list of changes, and then we gotta act based on the FileStatus. If it's DELETED, then we need to remove it from the right in case it was there.
        // for additions, it will be in the list of changes, and we know the content.
        // in each case, including modified, we also need to put the mode bits reflecting the ones in the working directory.

        // hmm, when a file is renamed _and_ edited with only partial sections selected, IJ will only have it in the list of partial changes with the new name.
        // but there is no corresponding change identifying the rename. are we not doing enough with the API?
        // ok so the original list of Changes does have that information.
        // ok, attach the original Change to the PartialFileChange so we can get it later.

        // Don't delete the directory itself to avoid any "open fd" issues when interacting with jj.
        // I don't know if these would actually happen. Just being extra pessimistic.
        Path(rightPath).listDirectoryEntries().forEach {
            FileUtil.delete(it)
        }
        Path(leftPath).toFile().copyRecursively(Path(rightPath).toFile())

        // Process all full changes by using the working copy as the source of truth.
        // Wherever there are modifications, copy the entire content over.
        // TODO: We are letting lots of exceptions potentially happen.
        rootChanges.fullChanges.forEach { fullChange: FullFileChange ->
            val change = fullChange.change
            when (change.type) {
                Change.Type.MODIFICATION, Change.Type.NEW -> {
                    FileUtil.copy(
                        fullChange.virtualFile!!.toNioPath().toFile(),
                        Path(
                            rightPath,
                            getRelativePathFromRoot(rootChanges.root, change.afterRevision!!.file)!!
                        ).toFile()
                    )
                }

                Change.Type.MOVED -> {
                    Path(
                        rightPath,
                        getRelativePathFromRoot(rootChanges.root, change.beforeRevision!!.file)!!
                    ).toFile().deleteRecursively()
                    FileUtil.copy(
                        fullChange.virtualFile!!.toNioPath().toFile(),
                        Path(
                            rightPath,
                            getRelativePathFromRoot(rootChanges.root, change.afterRevision!!.file)!!
                        ).toFile()
                    )
                }

                Change.Type.DELETED -> {
                    Path(
                        rightPath,
                        getRelativePathFromRoot(rootChanges.root, change.beforeRevision!!.file)!!
                    ).toFile().deleteRecursively()
                }
            }
        }

        // Process partial changes.
        rootChanges.partialChanges.forEach { partialChange: PartialFileChange ->
            val change = partialChange.change
            val helper = partialChange.helper
            when (change.type) {
                Change.Type.MODIFICATION -> {
                    FileUtil.writeToFile(
                        Path(
                            rightPath,
                            getRelativePathFromRoot(rootChanges.root, change.afterRevision!!.file)!!
                        ).toFile(), helper.content
                    )
                }

                Change.Type.MOVED -> {
                    Path(
                        rightPath,
                        getRelativePathFromRoot(rootChanges.root, change.beforeRevision!!.file)!!
                    ).toFile().deleteRecursively()
                    FileUtil.writeToFile(
                        Path(
                            rightPath,
                            getRelativePathFromRoot(rootChanges.root, change.afterRevision!!.file)!!
                        ).toFile(), helper.content
                    )
                }

                Change.Type.DELETED, Change.Type.NEW -> {
                    throw VcsException("Unexpected partial change type: ${change.type}")
                }
            }
        }
    }

    private fun getRelativePathFromRoot(root: VirtualFile, filePath: FilePath): String? {
        val rootPath = root.path
        val fullPath = filePath.path

        return if (fullPath.startsWith(rootPath)) {
            val relativePath = fullPath.substring(rootPath.length)
            // Remove leading separator if present
            if (relativePath.startsWith("/") || relativePath.startsWith("\\")) {
                relativePath.substring(1)
            } else {
                relativePath
            }
        } else {
            null
        }
    }

    /**
     * Separate changes into partial and full changes, following Git's pattern.
     * Returns a pair of (partial changes with helpers, remaining full changes).
     */
    private fun separatePartialAndFullChanges(changes: List<Change>): Pair<List<PartialFileChange>, List<FullFileChange>> {
        val partialChanges = mutableListOf<PartialFileChange>()

        // Use PartialChangesUtil to identify and collect partial changes
        val remainingChanges = PartialChangesUtil.processPartialChanges(
            project,
            changes,
            false
        ) { partialChangesList: List<ChangeListChange>?, tracker: PartialLocalLineStatusTracker? ->
            if (tracker != null && tracker.hasPartialChangesToCommit()) {
                val changelistIds = partialChangesList!!.map { it.changeListId }.distinct()
                val virtualFile = PartialChangesUtil.getVirtualFile(partialChangesList[0].change)

                if (virtualFile != null) {
                    try {
                        // Start the transaction - get helper with content
                        val helper = tracker.handlePartialCommit(Side.LEFT, changelistIds, true)

                        partialChanges.add(
                            PartialFileChange(
                                change = partialChangesList[0].change,
                                virtualFile = virtualFile,
                                helper = helper
                            )
                        )

                        LOG.debug("Collected partial change for ${virtualFile.path} with ${helper.content.length} chars")
                        return@processPartialChanges true // We handled this change
                    } catch (e: Exception) {
                        LOG.warn("Failed to handle partial commit for ${virtualFile.path}", e)
                        return@processPartialChanges false // Let it be handled as full change
                    }
                }
            }
            false // Not handled, will be in remainingChanges
        }

        // Convert remaining changes to FullFileChange objects
        val fullChanges = remainingChanges.map { change ->
            val virtualFile = PartialChangesUtil.getVirtualFile(change)
            FullFileChange(change, virtualFile)
        }

        return Pair(partialChanges, fullChanges)
    }


    private fun refreshRepository(root: VirtualFile) {
        try {
            // Also try to update repository if it exists
            val repositoryManager = VcsRepositoryManager.getInstance(project)
            val repository = repositoryManager.getRepositoryForRoot(root)
            if (repository != null) {
                repository.update()
                LOG.debug("Updated repository for root: ${root.path}")
            }
            LOG.debug("Refreshed VFS for root: ${root.path}")
        } catch (e: Exception) {
            LOG.warn("Failed to refresh repository for root: ${root.path}", e)
            // Fallback: just refresh the VFS
            root.refresh(false, true)
        }
    }

    override fun scheduleMissingFileForDeletion(files: List<FilePath>): List<VcsException>? {
        // jj automatically handles file deletions when they're missing from working copy
        // No explicit scheduling needed
        return null
    }

    override fun scheduleUnversionedFilesForAddition(files: List<VirtualFile>): List<VcsException>? {
        // jj automatically tracks all files in the working copy
        // No explicit scheduling needed
        return null
    }

    override fun isRefreshAfterCommitNeeded(): Boolean = true
}