package com.selvejj

import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.execution.util.ExecUtil
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.project.Project
import com.intellij.openapi.vcs.FilePath
import com.intellij.openapi.vcs.changes.Change
import com.intellij.openapi.vcs.changes.ContentRevision
import com.intellij.openapi.vcs.diff.DiffProvider
import com.intellij.openapi.vcs.diff.ItemLatestState
import com.intellij.openapi.vcs.history.VcsRevisionNumber
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.vcsUtil.VcsUtil
import com.intellij.openapi.vfs.VfsUtilCore

class SelvejjDiffProvider(private val project: Project) : DiffProvider {

    companion object {
        private val LOG = logger<SelvejjDiffProvider>()
    }

    override fun getCurrentRevision(file: VirtualFile): VcsRevisionNumber? {
        val repositoryRoot = getRepositoryRoot(file) ?: return null
        val commitId = runJjCommand(repositoryRoot.path, "log", "-r", "@-", "-T", "commit_id", "--no-graph")
        return if (commitId != null) SelvejjRevisionNumber(commitId) else null
    }

    override fun getLastRevision(virtualFile: VirtualFile): ItemLatestState? {
        val rev = getCurrentRevision(virtualFile)
        return if (rev != null) {
            ItemLatestState(rev, virtualFile.exists(), true)
        } else null
    }

    override fun getLastRevision(filePath: FilePath): ItemLatestState? {
        val repositoryRoot = VcsUtil.getVcsRootFor(project, filePath) ?: return null
        val relativePath = filePath.path.removePrefix(repositoryRoot.path + "/")

        val commitId =
            runJjCommand(repositoryRoot.path, "log", "-r", "@-", relativePath, "-T", "commit_id", "--no-graph")
        return if (commitId != null) {
            val file = filePath.virtualFile
            ItemLatestState(SelvejjRevisionNumber(commitId), file?.exists() ?: false, filePath.isDirectory)
        } else null
    }

    override fun createFileContent(revisionNumber: VcsRevisionNumber, selectedFile: VirtualFile): ContentRevision? {
        val repositoryRoot = getRepositoryRoot(selectedFile) ?: return null
        val relativePath = VfsUtilCore.getRelativePath(selectedFile, repositoryRoot) ?: return null

        return SelvejjHistoricalContentRevision(
            filePath = VcsUtil.getFilePath(selectedFile),
            revisionNumber = revisionNumber,
            repositoryRoot = repositoryRoot.path,
            relativePath = relativePath
        )
    }

    override fun getLatestCommittedRevision(vcsRoot: VirtualFile): VcsRevisionNumber? {
        val commitId = runJjCommand(vcsRoot.path, "log", "-r", "@-", "-T", "commit_id", "--no-graph")
        return if (commitId != null) SelvejjRevisionNumber(commitId) else null
    }

    override fun canCompareWithWorkingDir(): Boolean {
        return true
    }

    override fun compareWithWorkingDir(fileOrDir: VirtualFile, revNum: VcsRevisionNumber): Collection<Change> {
        if (revNum !is SelvejjRevisionNumber) {
            LOG.warn("Expected SelvejjRevisionNumber, got ${revNum::class.simpleName}")
            return emptyList()
        }

        return listOf(
            Change(createFileContent(revNum, fileOrDir), SelvejjContentRevision(VcsUtil.getFilePath(fileOrDir)))
        )

//        // If no changes found for a file, still create a Change object so the diff viewer
//        // can display the content side-by-side (even if identical)
//        if (changes.isEmpty() && !filePath.isDirectory) {
//            val beforeRevision = createFileContent(revNum, fileOrDir)
//            return VcsDiffUtil.createChangesWithCurrentContentForFile(filePath, beforeRevision)
//        }
//
//        return changes
    }

    private fun getRepositoryRoot(file: VirtualFile): VirtualFile? {
        return VcsUtil.getVcsRootFor(project, file)
    }

    private fun runJjCommand(workingDir: String, vararg args: String): String? {
        val command = GeneralCommandLine("jj")
            .withParameters(args.toList())
            .withWorkDirectory(workingDir)

        val result = ExecUtil.execAndGetOutput(command, 10000)
        return if (result.checkSuccess(LOG)) result.stdout.trim() else null
    }
}