/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.dvcs.DvcsUtil
import com.intellij.openapi.Disposable
import com.intellij.openapi.application.ApplicationManager
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.project.Project
import com.intellij.openapi.vfs.LocalFileSystem
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.openapi.vfs.VirtualFileManager
import com.intellij.openapi.vfs.newvfs.BulkFileListener
import com.intellij.openapi.vfs.newvfs.events.VFileEvent
import com.intellij.util.ui.update.MergingUpdateQueue
import com.intellij.util.ui.update.Update
import com.intellij.vcsUtil.VcsUtil
import java.io.File

/**
 * Updater for Jujutsu repositories.
 * Watches for changes in the .jj/working_copy directory and updates the repository state
 * when the checkout file changes.
 */
class SelvejjRepositoryUpdater(
    private val project: Project,
    private val repository: SelvejjRepository
) : BulkFileListener, Disposable {

    companion object {
        private val LOG = logger<SelvejjRepositoryUpdater>()
        private const val QUEUE_UPDATE_NAME = "SelvejjRepositoryUpdate"
        private const val QUEUE_UPDATE_DELAY_MS = 300
    }

    private val updateQueue = MergingUpdateQueue(
        QUEUE_UPDATE_NAME,
        QUEUE_UPDATE_DELAY_MS,
        true,
        null,
        this,
        null,
        false
    ).apply {
        setRestartTimerOnAdd(true)
    }

    private val watchRequests: Set<LocalFileSystem.WatchRequest>?

    init {
        // Add root to watch
        val workingCopyDir = File(repository.getJjDir(), "working_copy")
        val opHeadsDir = File(File(File(repository.getJjDir(), "repo"), "op_heads"), "heads")
        watchRequests = if (workingCopyDir.exists()) {
            LocalFileSystem.getInstance().addRootsToWatch(setOf(workingCopyDir.path, opHeadsDir.path), true)
        } else {
            LOG.warn("Working copy directory does not exist: ${workingCopyDir.path}")
            null
        }

        DvcsUtil.ensureAllChildrenInVfs(
            VcsUtil.getVirtualFile(workingCopyDir)
        )
        DvcsUtil.ensureAllChildrenInVfs(
            VcsUtil.getVirtualFile(opHeadsDir)
        )
        // Subscribe to VFS events
        project.messageBus.connect(this).subscribe(VirtualFileManager.VFS_CHANGES, this)

        // Schedule periodic updates
//        val executor = AppExecutorUtil.getAppScheduledExecutorService()
//        executor.scheduleWithFixedDelay(
//            { queueRepositoryUpdate() },
//            1,
//            30,
//            TimeUnit.SECONDS
//        )
    }

    /**
     * Called when files are changed.
     * Queues an update if the changes are relevant to the repository.
     */
    override fun after(events: List<VFileEvent>) {
        if (project.isDisposed) return

        val relevantEvents = events.filter { event ->
            val file = event.file ?: return@filter false
            isRelevantFile(file)
        }

        if (relevantEvents.isNotEmpty()) {
            LOG.debug("Checkout file change detected, queueing repository update")
            queueRepositoryUpdate()
        }
    }

    /**
     * Checks if the given file is relevant to the repository.
     * A file is relevant if it's the checkout file.
     */
    private fun isRelevantFile(file: VirtualFile): Boolean {
        val path = file.path
        val rootPath = repository.getJjDir().path
        return path.startsWith(rootPath)
    }

    /**
     * Queues an update for the repository.
     */
    private fun queueRepositoryUpdate() {
        if (project.isDisposed) return

        updateQueue.queue(Update.create(repository) {
            if (!project.isDisposed) {
                ApplicationManager.getApplication().invokeLater {
                    if (!project.isDisposed) {
                        LOG.debug("Updating repository: ${repository.root.path}")
                        repository.update()
                    }
                }
            }
        })
    }

    /**
     * Disposes the updater.
     */
    override fun dispose() {
        // Remove VFS watch
        watchRequests?.let {
            LocalFileSystem.getInstance().removeWatchedRoots(it)
        }

        // Cleanup resources
        updateQueue.cancelAllUpdates()
    }
}