/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.execution.util.ExecUtil
import com.intellij.openapi.Disposable
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.project.Project
import com.intellij.openapi.vcs.FilePath
import com.intellij.openapi.vcs.FileStatus
import com.intellij.openapi.vcs.VcsKey
import com.intellij.openapi.vcs.changes.Change
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.util.CollectConsumer
import com.intellij.util.Consumer
import com.intellij.vcs.log.*
import com.intellij.vcs.log.impl.VcsChangesLazilyParsedDetails
import com.intellij.vcs.log.impl.VcsFileStatusInfo
import com.intellij.vcsUtil.VcsUtil
import kotlinx.datetime.Instant
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import java.awt.Color
import java.io.DataInput
import java.io.DataOutput

@Serializable
data class SelvejjUser(val name: String, val email: String, val timestamp: Instant)

@Serializable
data class SelvejjCommitDetails(
    val commit_id: String,
    val change_id: String,
    val author: SelvejjUser,
    val committer: SelvejjUser,
    val parents: List<String>,
    val description: String
)

@Serializable
data class SelvejjParsedChange(
    val source: String,
    val target: String,
    val status: String
)

@Serializable
data class SelvejjFullCommitDetails(
    val metadata: SelvejjCommitDetails,
    val changes: List<SelvejjParsedChange>
)

class SelvejjVcsLogProvider(private val project: Project) : VcsLogProvider {

    companion object {
        private val LOG = logger<SelvejjVcsLogProvider>()

        private var FULL_COMMIT_JSON_TEMPLATE = "json(self)"
    }


    override fun readFirstBlock(
        root: VirtualFile,
        requirements: VcsLogProvider.Requirements
    ): VcsLogProvider.DetailedLogData {
        try {
            val factory = project.getService(VcsLogObjectsFactory::class.java)
            val commits = loadCommitDetails(root, "@ | @-", FULL_COMMIT_JSON_TEMPLATE)

            val commitMetadata = commits.map { commit ->
                factory.createCommitMetadata(
                    factory.createHash(commit.commit_id),
                    commit.parents.map { factory.createHash(it) },
                    commit.committer.timestamp.toEpochMilliseconds(),
                    root,
                    commit.description.split("\n").firstOrNull().orEmpty(),
                    commit.author.name,
                    commit.author.email,
                    commit.description,
                    commit.committer.name,
                    commit.committer.email,
                    commit.author.timestamp.toEpochMilliseconds()
                )
            }

            return object : VcsLogProvider.DetailedLogData {
                override fun getCommits(): List<VcsCommitMetadata> = commitMetadata
                override fun getRefs(): Set<VcsRef> = emptySet()
            }
        } catch (e: Exception) {
            LOG.warn("Failed to read first block", e)
            return object : VcsLogProvider.DetailedLogData {
                override fun getCommits(): List<VcsCommitMetadata> = emptyList()
                override fun getRefs(): Set<VcsRef> = emptySet()
            }
        }
    }

    override fun readAllHashes(root: VirtualFile, consumer: Consumer<in TimedVcsCommit>): VcsLogProvider.LogData {
        try {
            val factory = project.getService(VcsLogObjectsFactory::class.java)
            val userConsumer = CollectConsumer<VcsUser>(mutableSetOf())

            val commits = loadCommits(root, "all()", FULL_COMMIT_JSON_TEMPLATE)

            commits.forEach { commit ->
                consumer.consume(
                    factory.createTimedCommit(
                        factory.createHash(commit.commit_id),
                        commit.parents.map { factory.createHash(it) },
                        commit.committer.timestamp.toEpochMilliseconds()
                    )
                )

                if (commit.author.email.isNotEmpty()) {
                    userConsumer.consume(factory.createUser(commit.author.name, commit.author.email))
                }
                if (commit.committer.email.isNotEmpty()) {
                    userConsumer.consume(factory.createUser(commit.committer.name, commit.committer.email))
                }
            }

            return object : VcsLogProvider.LogData {
                override fun getRefs(): Set<VcsRef> = emptySet()

                @Suppress("UNCHECKED_CAST")
                override fun getUsers(): Set<VcsUser> = userConsumer.result as Set<VcsUser>
            }
        } catch (e: Exception) {
            LOG.warn("Failed to read all hashes", e)
            return object : VcsLogProvider.LogData {
                override fun getRefs(): Set<VcsRef> = emptySet()
                override fun getUsers(): Set<VcsUser> = emptySet()
            }
        }
    }

    override fun readMetadata(root: VirtualFile, commits: List<String>, consumer: Consumer<in VcsCommitMetadata>) {
        try {
            val factory = project.getService(VcsLogObjectsFactory::class.java)
            val revsetExpression = commits.joinToString(" | ")

            val commitDetails = loadCommitDetails(root, revsetExpression, FULL_COMMIT_JSON_TEMPLATE)

            commitDetails.forEach { commit ->
                consumer.consume(
                    factory.createCommitMetadata(
                        factory.createHash(commit.commit_id),
                        commit.parents.map { factory.createHash(it) },
                        commit.committer.timestamp.toEpochMilliseconds(),
                        root,
                        commit.description.split("\n").firstOrNull().orEmpty(),
                        commit.author.name,
                        commit.author.email,
                        commit.description,
                        commit.committer.name,
                        commit.committer.email,
                        commit.author.timestamp.toEpochMilliseconds(),
                    )
                )
            }
        } catch (e: Exception) {
            LOG.warn("Failed to read metadata for commits: $commits", e)
        }
    }

    override fun readFullDetails(
        root: VirtualFile,
        commits: List<String>,
        consumer: Consumer<in VcsFullCommitDetails>
    ) {
        try {
            val factory = project.getService(VcsLogObjectsFactory::class.java)
            val revsetExpression = commits.joinToString(" | ")

            val fullTemplate = """
                '{' ++
                '"metadata":' ++ $FULL_COMMIT_JSON_TEMPLATE ++ ', ' ++
                '"changes": [' ++ diff.files().map(|f| 
                    '{"status": "' ++ f.status() ++ '", ' ++
                    '"source": "' ++ f.source().path() ++ '", ' ++
                    '"target": "' ++ f.target().path() ++ '"}'
                ).join(', ') ++ ']' ++
                '}'
            """.trimIndent()

            val fullDetails = loadFullCommitDetails(root, revsetExpression, fullTemplate)
            val changesParser = SelvejjChangesParser()

            fullDetails.forEach { detail ->
                val metadata = detail.metadata
                val changes = detail.changes.map { change ->
                    VcsFileStatusInfo(
                        when (change.status) {
                            "removed" -> Change.Type.DELETED
                            "added" -> Change.Type.NEW
                            "modified" -> Change.Type.MODIFICATION
                            "renamed" -> Change.Type.MOVED
                            else -> Change.Type.MODIFICATION
                        },
                        change.source,
                        change.target.ifEmpty { null }
                    )
                }

                val commitDetails = VcsChangesLazilyParsedDetails(
                    project,
                    factory.createHash(metadata.commit_id),
                    metadata.parents.map { factory.createHash(it) },
                    metadata.committer.timestamp.toEpochMilliseconds(),
                    root,
                    metadata.description.split("\n").firstOrNull().orEmpty(),
                    factory.createUser(metadata.author.name, metadata.author.email),
                    metadata.description,
                    factory.createUser(metadata.committer.name, metadata.committer.email),
                    metadata.author.timestamp.toEpochMilliseconds(),
                    listOf(changes),
                    changesParser
                )

                consumer.consume(commitDetails)
            }
        } catch (e: Exception) {
            LOG.warn("Failed to read full details for commits: $commits", e)
        }
    }

    private fun loadCommits(root: VirtualFile, revset: String, template: String): List<SelvejjCommitDetails> {
        val command = GeneralCommandLine("jj")
            .withParameters("log", "-R", root.path, "--no-graph", "-r", revset, "-T", "$template ++ \"\\n\"")

        val result = ExecUtil.execAndGetOutput(command, 30000)
        if (!result.checkSuccess(LOG)) {
            throw RuntimeException("jj log command failed: ${result.stderr}")
        }

        val json = Json { ignoreUnknownKeys = true }
        return result.stdout.lines()
            .filter { it.trim().isNotEmpty() }
            .mapNotNull { line ->
                try {
                    json.decodeFromString<SelvejjCommitDetails>(line.trim())
                } catch (e: Exception) {
                    LOG.warn("Failed to parse commit line: $line", e)
                    null
                }
            }
    }

    private fun loadCommitDetails(root: VirtualFile, revset: String, template: String): List<SelvejjCommitDetails> {
        val command = GeneralCommandLine("jj")
            .withParameters("log", "-R", root.path, "--no-graph", "-r", revset, "-T", "$template ++ \"\\n\"")

        val result = ExecUtil.execAndGetOutput(command, 30000)
        if (!result.checkSuccess(LOG)) {
            throw RuntimeException("jj log command failed: ${result.stderr}")
        }

        val json = Json { ignoreUnknownKeys = true }
        return result.stdout.lines()
            .filter { it.trim().isNotEmpty() }
            .mapNotNull { line ->
                try {
                    json.decodeFromString<SelvejjCommitDetails>(line.trim())
                } catch (e: Exception) {
                    LOG.warn("Failed to parse commit details line: $line", e)
                    null
                }
            }
    }

    private fun loadFullCommitDetails(
        root: VirtualFile,
        revset: String,
        template: String
    ): List<SelvejjFullCommitDetails> {
        val command = GeneralCommandLine("jj")
            .withParameters("log", "-R", root.path, "--no-graph", "-r", revset, "-T", "$template ++ \"\\n\"")

        val result = ExecUtil.execAndGetOutput(command, 30000)
        if (!result.checkSuccess(LOG)) {
            throw RuntimeException("jj log command failed: ${result.stderr}")
        }

        val json = Json { ignoreUnknownKeys = true }
        return result.stdout.lines()
            .filter { it.trim().isNotEmpty() }
            .mapNotNull { line ->
                try {
                    json.decodeFromString<SelvejjFullCommitDetails>(line.trim())
                } catch (e: Exception) {
                    LOG.warn("Failed to parse full commit details line: $line", e)
                    null
                }
            }
    }

    override fun subscribeToRootRefreshEvents(roots: Collection<VirtualFile>, refresher: VcsLogRefresher): Disposable {
        val connection = project.messageBus.connect()

        // Subscribe to repository change events
        connection.subscribe(SelvejjRepository.JJ_REPO_CHANGE, object : SelvejjRepositoryChangeListener {
            override fun repositoryChanged(repository: SelvejjRepository) {
                if (roots.contains(repository.root)) {
                    LOG.debug("Repository changed, refreshing log: ${repository.root.path}")
                    refresher.refresh(repository.root)
                }
            }
        })

        return connection
    }

    override fun getCurrentUser(root: VirtualFile): VcsUser? {
        try {
            val factory = project.getService(VcsLogObjectsFactory::class.java)

            val nameCommand = GeneralCommandLine("jj")
                .withParameters("config", "get", "user.name")
                .withWorkDirectory(root.path)
            val nameResult = ExecUtil.execAndGetOutput(nameCommand, 5000)
            val name = if (nameResult.checkSuccess(LOG)) nameResult.stdout.trim() else ""

            val emailCommand = GeneralCommandLine("jj")
                .withParameters("config", "get", "user.email")
                .withWorkDirectory(root.path)
            val emailResult = ExecUtil.execAndGetOutput(emailCommand, 5000)
            val email = if (emailResult.checkSuccess(LOG)) emailResult.stdout.trim() else ""

            return if (name.isNotEmpty() || email.isNotEmpty()) {
                factory.createUser(name, email)
            } else null
        } catch (e: Exception) {
            LOG.warn("Failed to get current user", e)
            return null
        }
    }

    override fun getCurrentBranch(root: VirtualFile): String? = null

    override fun getContainingBranches(root: VirtualFile, hash: Hash): Collection<String> = emptyList()

    override fun <T : Any> getPropertyValue(property: VcsLogProperties.VcsLogProperty<T>): T? {
        @Suppress("UNCHECKED_CAST")
        return when (property) {
            VcsLogProperties.LIGHTWEIGHT_BRANCHES,
            VcsLogProperties.SUPPORTS_LOG_DIRECTORY_HISTORY,
            VcsLogProperties.HAS_COMMITTER -> true as T

            else -> null
        }
    }

    override fun getReferenceManager(): VcsLogRefManager {
        return object : VcsLogRefManager {
            override fun getLabelsOrderComparator(): Comparator<VcsRef> = Comparator { _, _ -> 0 }
            override fun getBranchLayoutComparator(): Comparator<VcsRef> = Comparator { _, _ -> 0 }
            override fun groupForBranchFilter(refs: Collection<VcsRef>): List<RefGroup> = emptyList()
            override fun groupForTable(refs: Collection<VcsRef>, compact: Boolean, expanded: Boolean): List<RefGroup> =
                emptyList()

            override fun serialize(output: DataOutput, type: VcsRefType) {}
            override fun deserialize(input: DataInput): VcsRefType = object : VcsRefType {
                override fun getBackgroundColor(): Color = Color.GRAY
                override fun isBranch(): Boolean = false
            }

            override fun isFavorite(ref: VcsRef): Boolean = false
            override fun setFavorite(ref: VcsRef, favorite: Boolean) {}
        }
    }

    override fun getSupportedVcs(): VcsKey = SelvejjVcs.getKey()
}

class SelvejjChangesParser : VcsChangesLazilyParsedDetails.ChangesParser {
    override fun parseStatusInfo(
        project: Project,
        commit: VcsShortCommitDetails,
        changes: List<VcsFileStatusInfo?>,
        parentIndex: Int
    ): List<Change?>? {
        val repositoryRoot = commit.root.path
        val parentCommit = if (commit.parents.isNotEmpty()) commit.parents[parentIndex] else null

        return changes.map { info ->
            info?.let {
                when (it.type) {
                    Change.Type.MODIFICATION -> {
                        val beforePath = VcsUtil.getFilePath("$repositoryRoot/${it.firstPath}", false)
                        val afterPath = VcsUtil.getFilePath("$repositoryRoot/${it.secondPath!!}", false)
                        Change(
                            createHistoricalContentRevision(beforePath, parentCommit, repositoryRoot),
                            createHistoricalContentRevision(afterPath, commit.id, repositoryRoot),
                            FileStatus.MODIFIED
                        )
                    }

                    Change.Type.NEW -> {
                        val afterPath = VcsUtil.getFilePath("$repositoryRoot/${it.secondPath!!}", false)
                        Change(
                            null,
                            createHistoricalContentRevision(afterPath, commit.id, repositoryRoot),
                            FileStatus.ADDED
                        )
                    }

                    Change.Type.DELETED -> {
                        val beforePath = VcsUtil.getFilePath("$repositoryRoot/${it.firstPath}", false)
                        Change(
                            createHistoricalContentRevision(beforePath, parentCommit, repositoryRoot),
                            null,
                            FileStatus.DELETED
                        )
                    }

                    Change.Type.MOVED -> {
                        val beforePath = VcsUtil.getFilePath("$repositoryRoot/${it.firstPath}", false)
                        val afterPath = VcsUtil.getFilePath("$repositoryRoot/${it.secondPath!!}", false)
                        Change(
                            createHistoricalContentRevision(beforePath, parentCommit, repositoryRoot),
                            createHistoricalContentRevision(afterPath, commit.id, repositoryRoot),
                            FileStatus.MODIFIED
                        )
                    }
                }
            }
        }
    }

    private fun createHistoricalContentRevision(
        filePath: FilePath,
        revisionHash: Hash?,
        repositoryRoot: String
    ): SelvejjHistoricalContentRevision? {
        if (revisionHash == null) return null

        val relativePath = filePath.path.removePrefix("$repositoryRoot/")
        return SelvejjHistoricalContentRevision(
            filePath = filePath,
            revisionNumber = SelvejjRevisionNumber(revisionHash.asString()),
            repositoryRoot = repositoryRoot,
            relativePath = relativePath
        )
    }
}