/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj.actions

import com.intellij.execution.configurations.GeneralCommandLine
import com.intellij.execution.util.ExecUtil
import com.intellij.openapi.actionSystem.ActionUpdateThread
import com.intellij.openapi.actionSystem.AnActionEvent
import com.intellij.openapi.application.ApplicationManager
import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.ide.CopyPasteManager
import com.intellij.openapi.progress.ProgressIndicator
import com.intellij.openapi.progress.ProgressManager
import com.intellij.openapi.progress.Task
import com.intellij.openapi.project.DumbAwareAction
import com.intellij.openapi.util.text.StringUtil
import com.intellij.util.containers.ContainerUtil
import com.intellij.vcs.log.VcsLogDataKeys
import com.selvejj.SelvejjVcs
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import java.awt.datatransfer.StringSelection

@Serializable
data class CommitToChangeMapping(
    val commit_id: String,
    val change_id: String
)

/**
 * Custom copy action that uses jj command to map commit IDs to change IDs
 * before copying to clipboard. This replaces the default "Copy Revision Number" action in VCS log.
 */
class CopyJjChangeIdAction : DumbAwareAction("Copy Change ID") {

    companion object {
        private val LOG = logger<CopyJjChangeIdAction>()

        // JSON template to extract both commit_id and change_id
        private val COMMIT_TO_CHANGE_TEMPLATE = """
            '{' ++
            '"commit_id": "' ++ commit_id ++ '", ' ++
            '"change_id": "' ++ change_id ++ '"' ++
            '}'
        """.trimIndent()
    }

    override fun actionPerformed(e: AnActionEvent) {
        val selection = e.getData(VcsLogDataKeys.VCS_LOG_COMMIT_SELECTION) ?: return
        val commits = ContainerUtil.reverse(selection.commits) // old to new order
        val project = e.project ?: return

        // TODO(nikhilm): This is obviously suboptimal. It would be better, given that we get VcsRevisionNumber classes and not strings,
        // if we could thread the change id into SelvejjRevisionNumber, and then down cast here and get what we want.
        ProgressManager.getInstance().run(object : Task.Backgroundable(project, "Getting Change IDs", false) {
            override fun run(indicator: ProgressIndicator) {
                // Group commits by repository root
                val commitsByRoot = commits.groupBy { it.root }
                val allChangeIds = mutableListOf<String>()

                for ((root, rootCommits) in commitsByRoot) {
                    val repositoryRoot = root.path
                    val commitIds = rootCommits.map { it.hash.asString() }
                    val changeIds = getChangeIdsFromCommitIds(repositoryRoot, commitIds)
                    allChangeIds.addAll(changeIds)
                }

                ApplicationManager.getApplication().invokeLater {
                    val copyText = StringUtil.join(allChangeIds, " ")
                    CopyPasteManager.getInstance().setContents(StringSelection(copyText))
                }
            }
        })
    }

    /**
     * Maps multiple commit IDs to their corresponding change IDs using a single jj command
     */
    private fun getChangeIdsFromCommitIds(repositoryRoot: String, commitIds: List<String>): List<String> {
        if (commitIds.isEmpty()) return emptyList()

        val revsetExpression = commitIds.joinToString(" | ")
        val command = GeneralCommandLine("jj")
            .withParameters(
                "log",
                "-R",
                repositoryRoot,
                "--no-graph",
                "-r",
                revsetExpression,
                "-T",
                "$COMMIT_TO_CHANGE_TEMPLATE ++ \"\\n\""
            )

        val result = ExecUtil.execAndGetOutput(command, 10000)
        if (!result.checkSuccess(LOG)) {
            LOG.warn("Failed to get change IDs for commits $commitIds: ${result.stderr}")
            return emptyList()
        }

        val json = Json { ignoreUnknownKeys = true }
        val mappings = result.stdout.lines()
            .filter { it.trim().isNotEmpty() }
            .mapNotNull { line ->
                try {
                    json.decodeFromString<CommitToChangeMapping>(line.trim())
                } catch (e: Exception) {
                    LOG.warn("Failed to parse commit to change mapping: $line", e)
                    null
                }
            }

        // Return change IDs in the same order as the input commit IDs
        return commitIds.mapNotNull { commitId ->
            mappings.find { it.commit_id == commitId }?.change_id
        }
    }

    override fun update(e: AnActionEvent) {
        val selection = e.getData(VcsLogDataKeys.VCS_LOG_COMMIT_SELECTION)
        val vcsLog = e.getData(VcsLogDataKeys.VCS_LOG)

        // Only enable for jj repositories with selected commits
        val isJjRepository = vcsLog?.logProviders?.values?.any { it.supportedVcs == SelvejjVcs.getKey() } == true
        val hasCommits = selection != null && selection.commits.isNotEmpty()
        e.presentation.isEnabled = hasCommits && isJjRepository

        // Update text to be more specific
        e.presentation.text = "Copy Change ID"
    }

    override fun getActionUpdateThread(): ActionUpdateThread {
        return ActionUpdateThread.BGT
    }
}