/*
 * Copyright (c) 2025 Nikhil Marathe <nikhil@selvejj.com>
 */

package com.selvejj

import com.intellij.driver.client.Remote
import com.intellij.driver.sdk.*
import com.intellij.ide.starter.driver.engine.runIdeWithDriver
import com.intellij.ide.starter.ide.IdeProductProvider
import com.intellij.ide.starter.models.TestCase
import com.intellij.ide.starter.plugins.PluginConfigurator
import com.intellij.ide.starter.project.LocalProjectInfo
import com.intellij.ide.starter.runner.Starter
import org.junit.jupiter.api.Assertions.assertEquals
import org.junit.jupiter.api.Assertions.assertNotNull
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.io.TempDir
import java.nio.file.Path
import kotlin.io.path.createDirectories
import kotlin.io.path.writeText
import kotlin.time.Duration.Companion.seconds

/**
 * Remote stub for VcsManager to verify VCS detection.
 */
@Remote("com.intellij.openapi.vcs.ProjectLevelVcsManager")
interface VcsManagerStub {
    fun getAllVcsRoots(): Array<VcsRootStub>
}

/**
 * Remote stub for VcsRoot to access VCS information.
 */
@Remote("com.intellij.openapi.vcs.VcsRoot")
interface VcsRootStub {
    fun getVcs(): VcsStub?
}

/**
 * Remote stub for AbstractVcs to get VCS name.
 */
@Remote("com.intellij.openapi.vcs.AbstractVcs")
interface VcsStub {
    fun getName(): String
}

/**
 * Integration tests for Selvejj VCS detection.
 * These tests launch an actual IDE instance and verify VCS detection.
 *
 * Based on https://blog.jetbrains.com/platform/2025/02/integration-tests-for-plugin-developers-intro-dependencies-and-first-integration-test/
 */
class SelvejjVcsDetectionIntegrationTest {

    @TempDir
    lateinit var testProjectDir: Path

    @Test
    fun testSelvejjVcsDetection() {
        // Create a non-colocated jj repository to avoid Git VCS detection
        val projectDir = testProjectDir.resolve("jj-project").createDirectories()
        val initProcess = ProcessBuilder("jj", "git", "init", "--no-colocate")
            .directory(projectDir.toFile())
            .redirectErrorStream(true)
            .start()

        val exitCode = initProcess.waitFor()
        require(exitCode == 0) {
            val output = initProcess.inputStream.bufferedReader().readText()
            "jj git init failed with exit code $exitCode: $output"
        }

        // Verify .jj directory was created
        val jjDir = projectDir.resolve(".jj")
        require(jjDir.toFile().exists()) { ".jj directory was not created at $jjDir" }

        // Create a file so the project has content
        projectDir.resolve("README.md").writeText("# Test Project")

        // Get plugin archive path from system property set by Gradle
        val pathToPlugin = System.getProperty("path.to.build.plugin")
        requireNotNull(pathToPlugin) { "path.to.build.plugin system property not set" }

        Starter.newContext(
            testName = "selvejjVcsDetection",
            TestCase(IdeProductProvider.IC, LocalProjectInfo(projectDir))
                .withVersion("2024.2")
        ).apply {
            PluginConfigurator(this).installPluginFromPath(Path.of(pathToPlugin))
        }.runIdeWithDriver().useDriverAndCloseIde {
            waitForProjectOpen()
            val project = singleProject()

            // Verify plugin is loaded
            val plugin = getPlugin("com.selvejj.selvejj")
            assertNotNull(plugin, "Selvejj plugin should be loaded")
            assertEquals(true, plugin.isEnabled(), "Selvejj plugin should be enabled")

            // Wait for VCS detection to complete
            val vcsManager = service(VcsManagerStub::class, project)
            waitFor(duration = 30.seconds, errorMessage = "VCS detection did not complete in time") {
                vcsManager.getAllVcsRoots().isNotEmpty()
            }

            // Verify Jujutsu VCS is detected
            val vcsRoots = vcsManager.getAllVcsRoots()
            val jjRoot = vcsRoots.find { it.getVcs()?.getName() == "Jujutsu" }
            assertNotNull(jjRoot, "Jujutsu VCS should be detected")
        }
    }
}
